package view;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.EventQueue;
import java.awt.Graphics;
import java.awt.Image;
import java.awt.image.BufferedImage;
import java.awt.print.PageFormat;
import java.awt.print.Printable;
import java.awt.print.PrinterException;
import java.awt.print.PrinterJob;

import javax.print.attribute.HashPrintRequestAttributeSet;
import javax.print.attribute.PrintRequestAttributeSet;
import javax.print.attribute.standard.JobName;
import javax.swing.JPanel;
import javax.swing.SwingUtilities;

import model.Hamster;
import model.Territorium;
import util.ImageLoader;
import util.Observer;

public class TerritoriumPanel extends JPanel implements Observer {

	private final static int CELLSIZE = 33;

	private Territorium territorium;
	private Image cornImage;
	private Image twoCornImage;
	private Image threeCornImage;
	private Image fourCornImage;
	private Image fiveCornImage;
	private Image sixCornImage;
	private Image sevenCornImage;
	private Image eightCornImage;
	private Image nineCornImage;
	private Image tenCornImage;
	private Image elevenCornImage;
	private Image twelveCornImage;
	private Image wallImage;
	private Image hamsterNorthImage;
	private Image hamsterWestImage;
	private Image hamsterSouthImage;
	private Image hamsterEastImage;
	
	private Dimension dimension;

	private Image territoryImage;
	final private Color territoryColor = new Color(255, 255, 220);

	public TerritoriumPanel(Territorium territorium) {
		super();
		this.territorium = territorium;
		this.territorium.addObserver(this);
		this.setBackground(this.territoryColor);
		this.loadImages();
		this.territoryImage = null;
		this.dimension = this.calcPreferredSize();
		this.setPreferredSize(this.dimension);
	}

	@Override
	public Dimension getPreferredSize() {
		return new Dimension((this.territorium.getNoOfCols() + 2)
				* TerritoriumPanel.CELLSIZE,
				(this.territorium.getNoOfRows() + 2)
						* TerritoriumPanel.CELLSIZE);
	}

	@Override
	public void paintComponent(Graphics g) {
		super.paintComponent(g);
		if (this.territoryImage == null) {
			createTerritoryImage();
		}
		int x = this.getWidth() / 2 - this.territorium.getNoOfCols()
				* TerritoriumPanel.CELLSIZE / 2;
		int y = this.getHeight() / 2 - this.territorium.getNoOfRows()
				* TerritoriumPanel.CELLSIZE / 2;
		g.drawImage(this.territoryImage, x, y, null);

	}

	public Image getTerritoryImage() {
		return this.territoryImage;
	}

	public void createTerritoryImage() {
		int width = this.territorium.getNoOfCols() * TerritoriumPanel.CELLSIZE
				+ 1;
		int height = this.territorium.getNoOfRows() * TerritoriumPanel.CELLSIZE
				+ 1;
		if (this.territoryImage == null) {
			this.territoryImage = new BufferedImage(width, height,
					BufferedImage.TYPE_INT_ARGB);
		}
		Graphics g = this.territoryImage.getGraphics();
		g.setColor(this.territoryColor);
		g.fillRect(0, 0, width, height);
		g.setColor(Color.BLACK);

		int x = 0;
		int y = 0;

		for (int r = 0; r <= this.territorium.getNoOfRows(); r++) {
			g.drawLine(x, y + r * TerritoriumPanel.CELLSIZE, x
					+ this.territorium.getNoOfCols()
					* TerritoriumPanel.CELLSIZE, y + r
					* TerritoriumPanel.CELLSIZE);
		}
		for (int c = 0; c <= this.territorium.getNoOfCols(); c++) {
			g.drawLine(x + c * TerritoriumPanel.CELLSIZE, y, x + c
					* TerritoriumPanel.CELLSIZE,
					y + this.territorium.getNoOfRows()
							* TerritoriumPanel.CELLSIZE);
		}

		for (int r = 0; r < this.territorium.getNoOfRows(); r++) {
			for (int c = 0; c < this.territorium.getNoOfCols(); c++) {
				if (this.territorium.isWallOnTile(r, c)) {
					g.drawImage(this.wallImage, x + c
							* TerritoriumPanel.CELLSIZE + 1, y + r
							* TerritoriumPanel.CELLSIZE + 1, null);
				} else {
					if (this.territorium.getNoOfGrains(r, c) > 0) {
						
						int numberOfGrains = this.territorium.getNoOfGrains(r, c);
						
						switch (numberOfGrains) {
						
							case 1:
								
								g.drawImage(this.cornImage, x + c
										* TerritoriumPanel.CELLSIZE + 1, y + r
										* TerritoriumPanel.CELLSIZE + 1, null);
								
								break;

							case 2:
								
								g.drawImage(this.twoCornImage, x + c
										* TerritoriumPanel.CELLSIZE + 1, y + r
										* TerritoriumPanel.CELLSIZE + 1, null);
								
								break;
								
							case 3:
								
								g.drawImage(this.threeCornImage, x + c
										* TerritoriumPanel.CELLSIZE + 1, y + r
										* TerritoriumPanel.CELLSIZE + 1, null);
								
								break;
								
							case 4:
								
								g.drawImage(this.fourCornImage, x + c
										* TerritoriumPanel.CELLSIZE + 1, y + r
										* TerritoriumPanel.CELLSIZE + 1, null);
								
								break;
								
							case 5:
								
								g.drawImage(this.fiveCornImage, x + c
										* TerritoriumPanel.CELLSIZE + 1, y + r
										* TerritoriumPanel.CELLSIZE + 1, null);
								
								break;
								
							case 6:
								
								g.drawImage(this.sixCornImage, x + c
										* TerritoriumPanel.CELLSIZE + 1, y + r
										* TerritoriumPanel.CELLSIZE + 1, null);
								
								break;
								
							case 7:
								
								g.drawImage(this.sevenCornImage, x + c
										* TerritoriumPanel.CELLSIZE + 1, y + r
										* TerritoriumPanel.CELLSIZE + 1, null);
								
								break;
								
							case 8:
								
								g.drawImage(this.eightCornImage, x + c
										* TerritoriumPanel.CELLSIZE + 1, y + r
										* TerritoriumPanel.CELLSIZE + 1, null);
								
								break;
								
							case 9:
								
								g.drawImage(this.nineCornImage, x + c
										* TerritoriumPanel.CELLSIZE + 1, y + r
										* TerritoriumPanel.CELLSIZE + 1, null);
								
								break;
								
							case 10:
								
								g.drawImage(this.tenCornImage, x + c
										* TerritoriumPanel.CELLSIZE + 1, y + r
										* TerritoriumPanel.CELLSIZE + 1, null);
								
								break;
								
							case 11:
								
								g.drawImage(this.elevenCornImage, x + c
										* TerritoriumPanel.CELLSIZE + 1, y + r
										* TerritoriumPanel.CELLSIZE + 1, null);
								
								break;
								
							default:
								
								g.drawImage(this.twelveCornImage, x + c
										* TerritoriumPanel.CELLSIZE + 1, y + r
										* TerritoriumPanel.CELLSIZE + 1, null);
								
								break;
						}
					}
					if (this.territorium.isHamsterOnTile(r, c)) {
						switch (this.territorium.getHamsterDirection()) {
						case Territorium.NORD:
							g.drawImage(this.hamsterNorthImage, x + c
									* TerritoriumPanel.CELLSIZE + 1, y + r
									* TerritoriumPanel.CELLSIZE + 1, null);
							break;
						case Territorium.WEST:
							g.drawImage(this.hamsterWestImage, x + c
									* TerritoriumPanel.CELLSIZE + 1, y + r
									* TerritoriumPanel.CELLSIZE + 1, null);
							break;
						case Territorium.SUED:
							g.drawImage(this.hamsterSouthImage, x + c
									* TerritoriumPanel.CELLSIZE + 1, y + r
									* TerritoriumPanel.CELLSIZE + 1, null);
							break;
						default: // case Hamster.OST:
							g.drawImage(this.hamsterEastImage, x + c
									* TerritoriumPanel.CELLSIZE + 1, y + r
									* TerritoriumPanel.CELLSIZE + 1, null);
							break;
						}
					}

				}
			}
		}
	}

	public Tile getTile(int x, int y) {
		int ox = this.getWidth() / 2 - this.territorium.getNoOfCols()
				* TerritoriumPanel.CELLSIZE / 2;
		int oy = this.getHeight() / 2 - this.territorium.getNoOfRows()
				* TerritoriumPanel.CELLSIZE / 2;
		if (x < ox
				|| y < oy
				|| x >= ox + this.territorium.getNoOfCols()
						* TerritoriumPanel.CELLSIZE
				|| y >= oy + this.territorium.getNoOfRows()
						* TerritoriumPanel.CELLSIZE) {
			return null;
		}

		int row = (y - oy) / TerritoriumPanel.CELLSIZE;
		int col = (x - ox) / TerritoriumPanel.CELLSIZE;
		return new Tile(row, col);
	}

	public Hamster getHamster(int x, int y) {
		Tile tile = this.getTile(x, y);
		if (tile != null && this.territorium.getHamsterCol() == tile.getCol()
				&& this.territorium.getHamsterRow() == tile.getRow()) {
			return this.territorium.getHamster();
		} else {
			return null;
		}
	}

	private void loadImages() {
		this.cornImage = ImageLoader.getImage("Corn32.png");
		this.twoCornImage = ImageLoader.getImage("TwoCorn32.png");
		this.threeCornImage = ImageLoader.getImage("ThreeCorn32.png");
		this.fourCornImage = ImageLoader.getImage("FourCorn32.png");
		this.fiveCornImage = ImageLoader.getImage("FiveCorn32.png");
		this.sixCornImage = ImageLoader.getImage("SixCorn32.png");
		this.sevenCornImage = ImageLoader.getImage("SevenCorn32.png");
		this.eightCornImage = ImageLoader.getImage("EightCorn32.png");
		this.nineCornImage = ImageLoader.getImage("NineCorn32.png");
		this.tenCornImage = ImageLoader.getImage("TenCorn32.png");
		this.elevenCornImage = ImageLoader.getImage("ElevenCorn32.png");
		this.twelveCornImage = ImageLoader.getImage("TwelveCorn32.png");
		this.wallImage = ImageLoader.getImage("Wall32.png");
		this.hamsterNorthImage = ImageLoader.getImage("hamsternorth.png");
		this.hamsterWestImage = ImageLoader.getImage("hamsterwest.png");
		this.hamsterSouthImage = ImageLoader.getImage("hamstersouth.png");
		this.hamsterEastImage = ImageLoader.getImage("hamstereast.png");
	}

	@Override
	public void update() {
		if (EventQueue.isDispatchThread()) { // bspw. TerritoriumDesigner
			this.checkPreferredSize();
			this.createTerritoryImage();
			this.repaint();
		} else { // Simulation-Thread
			SwingUtilities.invokeLater(new Runnable() {
				@Override
				public void run() {
					TerritoriumPanel.this.checkPreferredSize();
					TerritoriumPanel.this.createTerritoryImage();
					TerritoriumPanel.this.repaint();
				}
			});
		}
	}

	private void checkPreferredSize() {
		Dimension dim = this.calcPreferredSize();
		if (!dim.equals(this.dimension)) {
			if (this.territoryImage != null) {
				this.territoryImage.getGraphics().dispose();
				this.territoryImage = null;
			}
			this.dimension = dim;
			this.setSize(this.dimension); // damit u.U. die Scrollbars
											// erscheinen
			this.setPreferredSize(this.dimension);
		}

	}

	private Dimension calcPreferredSize() {
		return new Dimension((this.territorium.getNoOfCols() + 2)
				* TerritoriumPanel.CELLSIZE,
				(this.territorium.getNoOfRows() + 2)
						* TerritoriumPanel.CELLSIZE);
	}

	public boolean print() {
		try {
			PrinterJob pj = PrinterJob.getPrinterJob();
			pj.setPrintable(new TerritoryPrint());
			PrintRequestAttributeSet aset = new HashPrintRequestAttributeSet();
			aset.add(new JobName("Hamster-Territorium", null));
			if (pj.printDialog(aset)) {
				pj.print(aset);
			}
			return true;
		} catch (Exception exc) {
			exc.printStackTrace();
			return false;
		}
	}
	
	public boolean print2() {
		PrintRequestAttributeSet aset = new HashPrintRequestAttributeSet();
		PrinterJob pj = PrinterJob.getPrinterJob();
		pj.setJobName("hallo");
		pj.setPrintable(new TerritoryPrint());
		if (pj.printDialog(aset)) {
			try {
				pj.print(aset);
			} catch (PrinterException e1) {
				e1.printStackTrace();
			}
		}
		return true;
	}

	class TerritoryPrint implements Printable {
		int NOOF_PAGES = 1;

		@Override
		public int print(Graphics g, PageFormat pf, int pageIndex)
				throws PrinterException {
			if (pageIndex >= this.NOOF_PAGES) {
				return Printable.NO_SUCH_PAGE;
			}
			Image img = TerritoriumPanel.this.getTerritoryImage();
			double px = pf.getImageableX();
			double py = pf.getImageableY();
			double pw = pf.getImageableWidth();
			double ph = pf.getImageableHeight();
			double iw = img.getWidth(null);
			double ih = img.getHeight(null);
			int w = 0;
			int h = 0;
			if (pw / iw < ph / ih) {
				w = (int) pw;
				h = (int) (ih * pw / iw);
			} else {
				h = (int) ph;
				w = (int) (iw * ph / ih);
			}
			int x = (int) px + (int) (pw / 2) - w / 2;
			int y = (int) py + (int) (ph / 2) - h / 2;

			// druckt die Propertionen beibehaltend mittig und seitenf�llend
			g.drawImage(TerritoriumPanel.this.getTerritoryImage(), x, y, w, h,
					null);
			return Printable.PAGE_EXISTS;
		}
	}

}
