package tdd.controllers;



import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.List;
import java.util.Objects;

import javax.swing.JComponent;
import javax.tools.Diagnostic;
import javax.tools.JavaFileObject;

import tdd.interfaces.CompiledState;
import tdd.interfaces.ICompileErrorFixer;
import tdd.interfaces.ICompilerManager;
import tdd.interfaces.IProgramManager;
import tdd.interfaces.SimulationState;
import tdd.models.CustomTestContent;
import tdd.models.FinalStateTestContent;
import tdd.models.Template;
import tdd.models.TerritoryCheckpointTestContent;
import tdd.models.TerritorySequenceTestContent;
import tdd.models.TerritoryStates;
import tdd.models.Test;
import tdd.models.TestClassesDirectory;
import tdd.models.TestContent;
import tdd.models.TestContentViewContainer;
import tdd.models.Testsuite;
import tdd.models.TestsuiteContainer;
import tdd.templates.Templates;
import tdd.utils.Files;
import tdd.utils.JComponentFactory;
import tdd.utils.JComponentViewChooser;
import tdd.utils.Streams;
import tdd.views.CustomTestContentView;
import tdd.views.FinalStateTestContentView;
import tdd.views.TerritoryCheckpointTestContentView;
import tdd.views.TerritorySequenceTestContentView;
import util.Observer;



public class TestsuiteController {
	
	
	
	public static final String TEST_FILE_SUFFIX = ".test";

	
	
	private ICompilerManager compilerManager;
	private IProgramManager programManager;
	private TestsuiteContainer testsuiteContainer;
	private TestClassesDirectory testClassesDirectory;
	private TestContentViewContainer testContentViewContainer;
	private JComponentViewChooser viewChooser = new JComponentViewChooser();
	private TerritoryStates territoryStates;
	private TerritoryStateController territoryStateController;
	private SimulationState simulationState;
	private CompiledState compiledState;
	
	
	
	public TestsuiteController(
			ICompilerManager compilerManager,
			IProgramManager programManager,
			TestsuiteContainer testsuiteContainer, 
			TestClassesDirectory testClassesDirectory,
			TestContentViewContainer testContentViewContainer,
			TerritoryStates territoryStates,
			TerritoryStateController territoryStateController,
			SimulationState simulationState,
			CompiledState compiledState) {
	
		this.compilerManager = compilerManager;
		this.programManager = programManager;
		this.testsuiteContainer = testsuiteContainer;
		this.testClassesDirectory = testClassesDirectory;
		this.testContentViewContainer = testContentViewContainer;
		this.territoryStates = territoryStates;
		this.territoryStateController = territoryStateController;
		this.simulationState = simulationState;
		this.compiledState = compiledState;
		
		this.viewChooser.map(CustomTestContent.class, this.customTestViewFactory);
		this.viewChooser.map(FinalStateTestContent.class, this.finalStateTestViewFactory);
		this.viewChooser.map(TerritorySequenceTestContent.class, this.territorySequenceTestViewFactory);
		this.viewChooser.map(TerritoryCheckpointTestContent.class, this.territoryCheckpointTestViewFactory);
	}
	
	
	
	public ICompilerManager getCompilerManager() {
		
		return this.compilerManager;
	}
	
	public IProgramManager getProgramManager() {
		
		return this.programManager;
	}
	
	public TestsuiteContainer getTestsuiteContainer() {
		
		return this.testsuiteContainer;
	}
	
	public TestClassesDirectory getTestClassesDirectory() {
		
		return this.testClassesDirectory;
	}
	
	public TestContentViewContainer getTestContentViewContainer() {
		
		return this.testContentViewContainer;
	}
	
	
	
	public void initializeTestsuite(File directory) {
		
		if(!directory.exists()) {
			
			if(!directory.mkdirs()) {
				
				//TODO throw exception
			}
			
		} else {
			
			if(!directory.isDirectory()) {
				
				//TODO throw exception
			}
		}
		
		Testsuite testsuite = new Testsuite(null, directory);
		this.loadTestsuite(testsuite);
		
		this.getTestsuiteContainer().setTestsuite(testsuite);
	}
	
	public void saveTestsuite() {
		
		this.saveTestsuite(this.getTestsuiteContainer().getTestsuite());
	}
	
	public void saveTestsuite(Testsuite testsuite) {
		
		if(testsuite == null) return;
		
		for(Testsuite t : testsuite.getTestsuites()) this.saveTestsuite(t);
		for(Test t : testsuite.getTests()) this.saveTest(t);
	}
	
	public boolean compileTestsuite() {
		
		Testsuite testsuite = this.getTestsuiteContainer().getTestsuite();
		
		if(testsuite != null) {
			
			return this.compileTestsuite(testsuite);
			
		} else {
			
			//TODO throw exception... keine Testsuite vorhanden
		}
		
		return false;
	}
	
	public boolean compileTestsuite(Testsuite testsuite) {
		
		if(testsuite == null) {
			
			//TODO throw exception... testsuite is null
		}
		
		for(Testsuite subTestsuite : testsuite.getTestsuites()) {
			
			if(!this.compileTestsuite(subTestsuite)) {
				
				return false;
			}
		}
		
		for(Test test : testsuite.getTests()) {
			
			if(!this.compileTest(test)) {
				
				return false;
			}
		}
		
		return true;
	}
	
	public void loadTestsuite(Testsuite testsuite) {
		
		if(!testsuite.getDirectory().exists()) return;
		
		for(File file : testsuite.getDirectory().listFiles()) {
			
			if(file.isDirectory()) {
				
				this.loadTestsuite(new Testsuite(testsuite, file));
				
			} else {
				
				this.loadTest(testsuite, file);
			}
		}
	}
	
	public void moveTestsuite(Testsuite destination, Testsuite testsuite) {
		
		File file = testsuite.getDirectory();
		
		if(testsuite.getTestsuite() != null) {
			
			testsuite.getTestsuite().remove(testsuite);
		}
		
		File newFile = new File(destination.getDirectory(), testsuite.getDirectory().getName());
		file.renameTo(newFile);
		
		this.loadTestsuite(new Testsuite(destination, newFile));

		this.compiledState.setCompiled(false);
	}
	
	public void createTestsuite(Testsuite parent, String name) {
		
		if(parent.existsTestsuite(name)) return;
		
		File directory = new File(parent.getDirectory(), name);
		
		if(!directory.exists()) {
			
			if(!directory.mkdir()) {
				
				//TODO throw exception
				
				return;
			}
		}
		
		new Testsuite(parent, directory);	
	}
	
	public void deleteTestsuite(Testsuite testsuite) {
		
		Testsuite parent = testsuite.getTestsuite();
		
		if(parent != null) {
			
			Files.delete(testsuite.getDirectory());
			parent.remove(testsuite);
		}
	}
	
	public void createTest(Testsuite parent, String name, TestContent content) {
		
		if(parent.existsTest(name)) return;
		
		content.addObserver(this.testContentObserver);
		Test test = new Test(parent, name, content);
		this.saveTest(test);
		
		this.compiledState.setCompiled(false);
		
		this.showTest(test);
	}
	
	public void moveTest(Testsuite destination, Test test) {
		
		File file = new File(test.getTestsuite().getDirectory(), test.getName() + TEST_FILE_SUFFIX);
		
		test.getTestsuite().remove(test);
		
		File newFile = new File(destination.getDirectory(), test.getName() + TEST_FILE_SUFFIX);
		file.renameTo(newFile);
		
		this.loadTest(destination, newFile);

		this.compiledState.setCompiled(false);
		
		this.getTestContentViewContainer().setTestContentView(null);
	}
	
	public void createCustomTest(Testsuite parent, String name) {
		
		this.createTest(parent, name, new CustomTestContent());
	}
	
	public void createFinalStateTest(Testsuite parent, String name) {
		
		this.createTest(parent, name, new FinalStateTestContent());
	}
	
	public void createTerritorySequenceTest(Testsuite parent, String name) {
		
		this.createTest(parent, name, new TerritorySequenceTestContent());
	}
	
	public void createTerritoryCheckpointTest(Testsuite parent, String name) {
		
		this.createTest(parent, name, new TerritoryCheckpointTestContent());
	}

	public void saveTest(Test test) {
		
		this.storeTest(test);
	}
	
	public boolean compileTest(Test test) {
		
		try {
			
			String source = test.generateCode();
			final String className = test.getName();
			final File diectory = test.getTestsuite().getDirectory();
			
			JavaFileObject javaFileObject = this.getCompilerManager().createJavaFileObject(source, className, diectory);
		
			return this.getCompilerManager().compile(
					javaFileObject, 
					new File[] {
							this.getTestClassesDirectory().getDirecory(),
							this.getProgramManager().getProgramDirectory()
						},
					new ICompileErrorFixer() {
						
						@Override
						public JavaFileObject fix(List<Diagnostic<? extends JavaFileObject>> diagnostics, JavaFileObject javaFileObject) {
					
							if(diagnostics.size() > 0) {
								
								Diagnostic<?> diagnostic = diagnostics.get(0);
								
								try {
									
									CharSequence charSource = javaFileObject.getCharContent(true);
									
									if(charSource != null) {
										
										String source = charSource.toString();
										
										if(
												Objects.equals("compiler.err.cant.apply.symbols", diagnostic.getCode()) || 
												Objects.equals("compiler.err.cant.resolve.location.args", diagnostic.getCode()) ||
												Objects.equals("compiler.err.cant.resolve.location", diagnostic.getCode())) {
											
											int diagnosticStartIndex = (int) diagnostic.getStartPosition();
											
											
											int startIndexFromSemicolon = source.substring(0, diagnosticStartIndex).lastIndexOf(';');
											int startIndexFromOpenCurlyBrace = source.substring(0, diagnosticStartIndex).lastIndexOf('{');
											int startIndexFromCloseCurlyBrace = source.substring(0, diagnosticStartIndex).lastIndexOf('}');
											
											int startIndex = startIndexFromSemicolon;
											
											if(startIndexFromOpenCurlyBrace > startIndex) startIndex = startIndexFromOpenCurlyBrace;
											if(startIndexFromCloseCurlyBrace > startIndex) startIndex = startIndexFromCloseCurlyBrace;
											
											
											int endIndexFromSemicolon = source.indexOf(';', diagnosticStartIndex);
											int endIndexFromOpenCurlyBrace = source.indexOf('{', diagnosticStartIndex) - 1;
											int endIndexFromCloseCurlyBrace = source.indexOf('}', diagnosticStartIndex);
											
											int endIndex = endIndexFromSemicolon;
											
											if(endIndexFromOpenCurlyBrace > 0 && (endIndex < 0 || endIndex > endIndexFromOpenCurlyBrace)) 
												endIndex = endIndexFromOpenCurlyBrace;
											
											if(endIndexFromCloseCurlyBrace > 0 && (endIndex < 0 || endIndex > endIndexFromCloseCurlyBrace)) 
												endIndex = endIndexFromCloseCurlyBrace;
											
											if(endIndex >= 0 && startIndex >= 0) {
												
												String newSource = source.substring(0, startIndex + 1);
												String methodCall = source.substring(diagnosticStartIndex, endIndex + 1);
												methodCall = methodCall.replace("\"", "\\\"");
												
												Template methodCallAssert = Templates.create(Streams.fromResources(Templates.MethodCallAssert.PATH));
												methodCallAssert.set(Templates.MethodCallAssert.VARIABLE_METHOD_CALL, methodCall);
												
												newSource += methodCallAssert.compile();
												newSource += source.substring(endIndex + 1, source.length());
												
												return getCompilerManager().createJavaFileObject(newSource, className, diectory);
											}
											
										} else if(Objects.equals("compiler.err.unreachable.stmt", diagnostic.getCode())) {
											
											int diagnosticStartIndex = (int) diagnostic.getStartPosition();
											int diagnosticEndIndex = (int) diagnostic.getEndPosition();
											
											String newSource = source.substring(0, diagnosticStartIndex);
											newSource += source.substring(diagnosticEndIndex, source.length());
											
											return getCompilerManager().createJavaFileObject(newSource, className, diectory);
										}
									}
									
								} catch (IOException e) {
						
									e.printStackTrace();
								}
								
								return null;
							}
		
							return javaFileObject;
						}
					}
				);
			
		} catch (IOException e) {
	
			e.printStackTrace();
			//TODO throw exception... cant create code
		}
		
		return false;
	}

	public void deleteTest(Test test) {
		
		Testsuite parent = test.getTestsuite();
		
		if(parent != null) {

			this.getTestContentViewContainer().setTestContentView(null);
			
			Files.delete(new File(parent.getDirectory(), test.getName() + TEST_FILE_SUFFIX));
			parent.remove(test);
		}
	}
	
	public void showTest(Test test) {
		
		this.getTestContentViewContainer().setTestContentView(this.viewChooser.create(test.getTestContent()));
	}
	
	
	
	private void storeTest(Test test) {
		
		FileOutputStream fos = null;
		ObjectOutputStream oos = null;
		
		try {
			
			File file = new File(test.getTestsuite().getDirectory(), test.getName() + TEST_FILE_SUFFIX);
			fos = new FileOutputStream(file);
			oos = new ObjectOutputStream(fos);
			
			oos.writeObject(test.getTestContent());
			
		} catch (Exception e) {
			
			//TODO throw exception
			return;
			
		} finally {
			
			if(oos != null) {
				
				try {
					
					oos.close();
					
				} catch (IOException e) {}
			}
			
			if(fos != null) {
				
				try {
					
					fos.close();
					
				} catch (IOException e) {}
			}
		}
	}
	
	private void loadTest(Testsuite testsuite, File file) {
		
		if(!file.getAbsolutePath().endsWith(TEST_FILE_SUFFIX)) return;
		
		FileInputStream fis = null;
		ObjectInputStream ois = null;
		
		try {
			
			fis = new FileInputStream(file); 
			ois = new ObjectInputStream(fis);
			
			TestContent content = (TestContent) ois.readObject();
			
			String name = file.getName();
			name = name.substring(0, name.length() - TEST_FILE_SUFFIX.length());
			
			new Test(testsuite, name, content);
			
			content.addObserver(this.testContentObserver);
			
		} catch(Exception e) {
			
			//TODO exception definieren
			return;
			
		} finally {
			
			if(ois != null) {
				
				try {
					
					ois.close();
					
				} catch(Exception e) {}
			}
			
			if(fis != null) {
				
				try {
					
					fis.close();
					
				} catch(Exception e) {}
			}
		}
	}
	
	
	
	private final Observer testContentObserver = new Observer() {
		
		@Override
		public void update() {
			
			compiledState.setCompiled(false);
		}
	};
	
	private final JComponentFactory customTestViewFactory = new JComponentFactory() {
		
		@Override
		public JComponent create(Object value) {
			
			if(!(value instanceof CustomTestContent)) return null;
			
			return new CustomTestContentView((CustomTestContent) value);
		}
	};
	
	private final JComponentFactory finalStateTestViewFactory = new JComponentFactory() {
		
		@Override
		public JComponent create(Object value) {
			
			if(!(value instanceof FinalStateTestContent)) return null;
			
			return new FinalStateTestContentView((FinalStateTestContent) value, territoryStates, territoryStateController);
		}
	};
	
	private final JComponentFactory territorySequenceTestViewFactory = new JComponentFactory() {
		
		@Override
		public JComponent create(Object value) {
			
			if(!(value instanceof TerritorySequenceTestContent)) return null;
			
			return new TerritorySequenceTestContentView((TerritorySequenceTestContent) value, territoryStates, territoryStateController, simulationState);
		}
	};
	
	private final JComponentFactory territoryCheckpointTestViewFactory = new JComponentFactory() {
		
		@Override
		public JComponent create(Object value) {
			
			if(!(value instanceof TerritoryCheckpointTestContent)) return null;
			
			return new TerritoryCheckpointTestContentView((TerritoryCheckpointTestContent) value, territoryStates, territoryStateController, simulationState);
		}
	};
}
