package tdd.controllers;



import java.io.File;
import java.lang.annotation.Annotation;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.List;

import javax.swing.JOptionPane;

import tdd.annotations.DefaultThrowable;
import tdd.annotations.Einrichtung;
import tdd.errors.ErwartungError;
import tdd.interfaces.CompiledState;
import tdd.interfaces.ICompilerManager;
import tdd.interfaces.IHamsterProgram;
import tdd.interfaces.IProgramManager;
import tdd.interfaces.ISimulationManager;
import tdd.interfaces.IStateManager;
import tdd.interfaces.SimulationState;
import tdd.models.TerritoryState;
import tdd.models.Test;
import tdd.models.TestCaseRunInformation;
import tdd.models.TestClassesDirectory;
import tdd.models.TestRunnerInformation;
import tdd.models.Testsuite;
import tdd.models.TestsuiteContainer;
import util.Observer;
import util.Resources;



public class TestRunner {


	
	private TestsuiteContainer testsuiteContainer;
	private TestRunnerInformation testRunnerInformation;
	private ICompilerManager compilerManager;
	private IProgramManager programManager;
	private ISimulationManager simulationManager;
	private IStateManager stateManager;
	private SimulationState simulationState;
	private TestClassesDirectory testClassesDirectory;
	private AsyncTestRunner asyncTestRunner;
	private TerritoryStateController territoryStateController;
	private CompiledState compiledState;
	
	
	
	public TestRunner(
			ICompilerManager compilerManager, 
			IProgramManager programManager, 
			ISimulationManager simulationManager,
			SimulationState simulationState,
			IStateManager stateManager,
			TestsuiteContainer testsuiteContainer,
			TestRunnerInformation information,
			TestClassesDirectory testClassesDirectory,
			TerritoryStateController territoryStateController,
			CompiledState compiledState) {
	
		this.compilerManager = compilerManager;
		this.programManager = programManager;
		this.simulationManager = simulationManager;
		this.simulationState = simulationState;
		this.stateManager = stateManager;
		this.testsuiteContainer = testsuiteContainer;
		this.testRunnerInformation = information;
		this.testClassesDirectory = testClassesDirectory;
		this.territoryStateController = territoryStateController;
		this.compiledState = compiledState;
	}	
	
	
	
	public ICompilerManager getCompilerManager() {
		
		return this.compilerManager;
	}
	
	public IProgramManager getProgramManager() {
		
		return this.programManager;
	}
	
	public ISimulationManager getSimulationManager() {
		
		return this.simulationManager;
	}
	
	public SimulationState getSimulationState() {
		
		return this.simulationState;
	}
	
	public IStateManager getStateManager() {
		
		return this.stateManager;
	}
	
	public TestsuiteContainer getTestsuiteContainer() {
		
		return this.testsuiteContainer;
	}
	
	public TestRunnerInformation getTestRunnerInformation() {
		
		return this.testRunnerInformation;
	}
	
	public TestClassesDirectory getTestClassesDirectory() {
		
		return this.testClassesDirectory;
	}
	
	public TerritoryStateController getTerritoryStateController() {
		
		return this.territoryStateController;
	}
	
	
	
	public void start() {
		
		Testsuite testsuite = this.getTestsuiteContainer().getTestsuite();
		
		if(testsuite != null) this.start(testsuite);
	}
	
	public void start(Testsuite testsuite) {
		
		this.run(testsuite);
	}
	
	public void start(Test test) {
		
		this.run(new Test[] {test});
	}
	
	
	
	private void run(Testsuite testsuite) {
		
		this.run(testsuite.getAllTests());
	}
	
	private void run(final Test[] tests) {

		if(!compiledState.isCompiled()) {
			
			boolean compiled = this.compilerManager.compile();
			
			if(compiled) {

				JOptionPane.showMessageDialog(null,
						Resources.getValue("compile.success"),
						Resources.getValue("compile.title"),
						JOptionPane.INFORMATION_MESSAGE);
			}
		}
		
		if(compiledState.isCompiled()) {
			
			synchronized (this) {
				
				if(this.asyncTestRunner == null) {
				
					this.getTestRunnerInformation().clear();
					
					this.asyncTestRunner = new AsyncTestRunner(tests, this.getSimulationState());
					this.asyncTestRunner.start();
				}
			}
		}
	}
	
	
	
	private class AsyncTestRunner extends Thread {
		
		
		
		private SimulationState simulationState;
		private Test[] tests;
		
		
		
		public AsyncTestRunner(Test[] tests, SimulationState simulationState) {
		
			this.tests = tests;
			this.simulationState = simulationState;
		}
		
		
		
		public Test[] getTests() {
			
			return this.tests;
		}
		
		public SimulationState getSimulationState() {
			
			return this.simulationState;
		}
		
		
		
		@Override
		public void run() {
		
			TerritoryState territoryState = new TerritoryState(TestRunner.this.getStateManager().getTerritoryState());
			
			for(Test test : this.getTests()) {
				
				Class<?> cls = TestRunner.this.getCompilerManager().loadClass(
						new File[] {
								test.getTestsuite().getDirectory(),
								TestRunner.this.getTestClassesDirectory().getDirecory(),
								TestRunner.this.getProgramManager().getProgramDirectory()}, 
						test.getName()
					);
				
				if(cls == null) return;
				
				TestRunner.this.getStateManager().resetHamsterTerritoryState();
				
				List<EinrichtungMethod> setupMethods = new ArrayList<EinrichtungMethod>();
				List<TestMethod> testCaseMethods = new ArrayList<TestMethod>();
				
				for(Method method : cls.getDeclaredMethods()) {
					
					Annotation annotation = method.getAnnotation(Einrichtung.class);
					
					if(annotation != null) {
						
						setupMethods.add(new EinrichtungMethod((Einrichtung) annotation, method));
					}
					
					annotation = method.getAnnotation(tdd.annotations.Test.class);
					
					if(annotation != null) {
						
						testCaseMethods.add(new TestMethod((tdd.annotations.Test) annotation, method));
					}
				}
				
				
				for(TestMethod testCaseMethod : testCaseMethods) {
				
					TestRunner.this.getStateManager().setTerritoryState(territoryState);
					TestRunner.this.getSimulationManager().sleep();
					
					try {
						
						Object testInstance = cls.getConstructor(TerritoryStateController.class).newInstance(TestRunner.this.getTerritoryStateController());
						
						TestCaseRun testCaseRun = new TestCaseRun(
								test,
								setupMethods.toArray(new EinrichtungMethod[setupMethods.size()]), 
								testCaseMethod, 
								testInstance
							);

						this.getSimulationState().addObserver(this.simulationStateObserver);
						
						TestRunner.this.getSimulationManager().run(testCaseRun);
						
						synchronized (this) {
							
							this.wait();
						}
						
					} catch (Exception e) {
						
						e.printStackTrace();
					}
					
					this.getSimulationState().deleteObserver(this.simulationStateObserver);
					
					if(TestRunner.this.getTestRunnerInformation().hasAnyError()) {
						
						break;
					}
				}
				
				if(TestRunner.this.getTestRunnerInformation().hasAnyError()) {
					
					break;
				}
			}
			
			if(!TestRunner.this.getTestRunnerInformation().hasAnyError()) {
				
				TestRunner.this.getStateManager().setTerritoryState(territoryState);
			}
			
			TestRunner.this.getCompilerManager().loadAndSetNewHamster();
			
			synchronized (TestRunner.this) {
				
				TestRunner.this.asyncTestRunner = null;
			}
		}
		
		
		
		private final Observer simulationStateObserver = new Observer() {
			
			@Override
			public void update() {
				
				synchronized (AsyncTestRunner.this) {
					
					if(!AsyncTestRunner.this.getSimulationState().isRunning()) {
						
						AsyncTestRunner.this.notify();
					}
				}
			}
		};
	}
	
	
	private class TestCaseRun implements IHamsterProgram {

		private Test test;
		private EinrichtungMethod[] setupMethods;
		private TestMethod testCaseMethod;
		private Object testInstance;

		public TestCaseRun(Test test, EinrichtungMethod[] setupMethods, TestMethod testCaseMethod, Object testInstance) {
			
			this.test = test;
			this.setupMethods = setupMethods;
			this.testCaseMethod = testCaseMethod;
			this.testInstance = testInstance;
		}
		
		@Override
		public void main() {
			
			TestRunnerInformation testRunnerInformation = TestRunner.this.getTestRunnerInformation();
			TestCaseRunInformation testCaseRunInformation = testRunnerInformation.add(
					this.test.getFullName(), 
					this.testCaseMethod.getMethod().getName(), 
					true
				);
			
			try {
				
				Method runSetupMethod = this.testInstance.getClass().getMethod("runSetupMethod", Method.class);
				Method runTestCaseMethod = this.testInstance.getClass().getMethod("runTestCaseMethod", Method.class);
				
				for(AnnotationMethod<Einrichtung> setupMethod : this.setupMethods) {
					
					runSetupMethod.invoke(this.testInstance, setupMethod.getMethod());
				}
				
				runTestCaseMethod.invoke(this.testInstance, this.testCaseMethod.getMethod());
				
				if(this.testCaseMethod.getAnnotation().erwarteFehler().equals(DefaultThrowable.class)) {
					
					testCaseRunInformation.setRunning(false);
					
				} else {
					
					testCaseRunInformation.setError(
							new ErwartungError(
									"Es wurde der Fehler " + this.testCaseMethod.getAnnotation().erwarteFehler().getName() + " erwartet", 
									null
								)
						);
				}
				
				testCaseRunInformation.setRunning(false);
				
			} catch(InvocationTargetException e) {
				
				InvocationTargetException innerException = (InvocationTargetException) e.getTargetException();
				Throwable t = innerException.getTargetException();
				
				if(t.getClass().equals(this.testCaseMethod.getAnnotation().erwarteFehler())) {
					
					testCaseRunInformation.setRunning(false);
					
				} else {
					
					testCaseRunInformation.setError(
							new ErwartungError(
									t.toString(), 
									null
								)
						);
				}
				
			} catch(Exception e) {
				
				e.printStackTrace();
				testCaseRunInformation.setError(new ErwartungError(e.toString(), null));
			}
		}
	}
	
	private class EinrichtungMethod extends AnnotationMethod<Einrichtung> {

		public EinrichtungMethod(Einrichtung annotation, Method method) {
			
			super(annotation, method);
		}
	}
	
	private class TestMethod extends AnnotationMethod<tdd.annotations.Test> {

		public TestMethod(tdd.annotations.Test annotation, Method method) {
			
			super(annotation, method);
		}
	}
	
	private class AnnotationMethod<TAnnotation extends Annotation> {
		
		private TAnnotation annotation;
		private Method method;
		
		
		public AnnotationMethod(TAnnotation annotation, Method method) {
		
			this.annotation = annotation;
			this.method = method;
		}
		
		public TAnnotation getAnnotation() {
			
			return this.annotation;
		}
		
		public Method getMethod() {
			
			return this.method;
		}
	}
}
