package model;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;

import javax.xml.namespace.QName;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;
import javax.xml.stream.XMLEventReader;
import javax.xml.stream.XMLInputFactory;
import javax.xml.stream.XMLOutputFactory;
import javax.xml.stream.XMLStreamConstants;
import javax.xml.stream.XMLStreamReader;
import javax.xml.stream.XMLStreamWriter;
import javax.xml.stream.events.Characters;
import javax.xml.stream.events.StartElement;
import javax.xml.stream.events.XMLEvent;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.DefaultHandler;

import tdd.interfaces.IHamster;
import tdd.interfaces.ITerritory;
import util.Observable;
import util.Observer;
import controller.program.CompileManager;

// Anmerkung:
// Synchronisation notwendig, um es zu erm�glichen, dass w�hrend eine Simulation l�uft noch der
// Zustand des Territoriums ge�ndert werden kann

public class Territorium extends Observable implements Serializable, ITerritory {

	public final static int NORD = Hamster.NORD;
	public final static int WEST = Hamster.WEST;
	public final static int SUED = Hamster.SUED;
	public final static int OST = Hamster.OST;

	final static int NOOF_ROWS = 10;
	final static int NOOF_COLS = 15;

	final static int WALL_ON_TILE = -1;

	int rows;
	int cols;
	int[][] tiles;
	transient Hamster hamster;

	public Territorium() {
		this(Territorium.NOOF_ROWS, Territorium.NOOF_COLS);
	}

	public Territorium(int rows, int cols) {
		this.rows = Territorium.NOOF_ROWS;
		this.cols = Territorium.NOOF_COLS;
		this.tiles = new int[this.rows][this.cols];
		for (int r = 0; r < this.rows; r++) {
			for (int c = 0; c < this.cols; c++) {
				this.tiles[r][c] = 0; // alle Kacheln sind leer
			}
		}
		this.hamster = new Hamster(this);
	}

	public synchronized Hamster getHamster() {
		return this.hamster;
	}

	public synchronized void changeHamster(Hamster ham) {
		for (Observer o : this.hamster.getObservers()) {
			ham.addObserver(o);
		}
		this.hamster.deleteObservers();
		this.hamster = ham;
		this.notifyObservers();
	}

	@Override
	public synchronized void addObserver(Observer o) {
		super.addObserver(o);
		this.hamster.addObserver(o);
	}

	@Override
	public void activateNotification() {
		super.activateNotification();
		this.hamster.activateNotification();
	}

	@Override
	public void deactivateNotification() {
		super.deactivateNotification();
		this.hamster.deactivateNotification();
	}

	public synchronized void setSize(int newRows, int newCols) {
		if (newRows == this.rows && newCols == this.cols) {
			return;
		}
		this.changeSize(newRows, newCols);
		if (this.hamster.hamsterRow >= newRows
				|| this.hamster.hamsterCol >= newCols) {
			// Hamster umplatzieren
			if (this.tiles[0][0] == Territorium.WALL_ON_TILE) {
				this.tiles[0][0] = 0; // Mauer entfernen
			}
			this.hamster.hamsterRow = 0;
			this.hamster.hamsterCol = 0;
		}
		this.notifyObservers();
	}

	void changeSize(int newRows, int newCols) {
		if (newRows == this.rows && newCols == this.cols) {
			return;
		}
		int[][] newTiles = new int[newRows][newCols];
		for (int r = 0; r < newRows; r++) {
			for (int c = 0; c < newCols; c++) {
				if (r < this.rows && c < this.cols) {
					newTiles[r][c] = this.tiles[r][c]; // uebernehmen
				} else {
					newTiles[r][c] = 0; // leer
				}
			}
		}
		this.tiles = newTiles;
		this.rows = newRows;
		this.cols = newCols;
	}

	public synchronized int getNoOfRows() {
		return this.rows;
	}

	public synchronized int getNoOfCols() {
		return this.cols;
	}

	public synchronized boolean isWallOnTile(int row, int col) {
		return this.tiles[row][col] == Territorium.WALL_ON_TILE;
	}

	public synchronized boolean isHamsterOnTile(int row, int col) {
		return this.hamster.hamsterRow == row && this.hamster.hamsterCol == col;
	}

	public synchronized void setHamsterOnTile(int row, int col) {
		if (!this.isWallOnTile(row, col)) {
			this.hamster.hamsterRow = row;
			this.hamster.hamsterCol = col;
			this.notifyObservers();
		}
	}

	public synchronized void setWall(int row, int col) {
		if (!(this.hamster.hamsterRow == row && this.hamster.hamsterCol == col)) {
			this.tiles[row][col] = Territorium.WALL_ON_TILE;
			this.notifyObservers();
		}
	}

	public synchronized void changeNoOfGrains(int row, int col, int number) {
		if (!(this.tiles[row][col] == Territorium.WALL_ON_TILE)) {
			this.tiles[row][col] += number;
			this.notifyObservers();
		}
	}

	public synchronized void setNoOfGrains(int row, int col, int number) {
		if (!(this.tiles[row][col] == Territorium.WALL_ON_TILE)) {
			this.tiles[row][col] = number;
			this.notifyObservers();
		}
	}

	public synchronized int getNoOfGrains(int row, int col) {
		if (this.tiles[row][col] == Territorium.WALL_ON_TILE) {
			return 0;
		} else {
			return this.tiles[row][col];
		}
	}

	public synchronized void clear(int row, int col) {
		this.tiles[row][col] = 0;
		this.notifyObservers();
	}

	public synchronized void clear() {
		for (int r = 0; r < this.rows; r++) {
			for (int c = 0; c < this.cols; c++) {
				this.tiles[r][c] = 0; // alle Kacheln sind leer
			}
		}
		this.notifyObservers();
	}

	public synchronized int getHamsterRow() {
		return this.hamster.hamsterRow;
	}

	public synchronized int getHamsterCol() {
		return this.hamster.hamsterCol;
	}

	public synchronized int getHamsterDirection() {
		return this.hamster.hamsterDirection;
	}

	public synchronized int getHamsterGrains() {
		return this.hamster.hamsterGrains;
	}

	public synchronized boolean serialize(String filename) {
		FileOutputStream fs = null;
		ObjectOutputStream os = null;
		try {
			fs = new FileOutputStream(filename);
			os = new ObjectOutputStream(fs);
			os.writeObject(this);
			os.writeInt(this.hamster.hamsterRow);
			os.writeInt(this.hamster.hamsterCol);
			os.writeInt(this.hamster.hamsterDirection);
			os.writeInt(this.hamster.hamsterGrains);
			return true;
		} catch (IOException e) {
			e.printStackTrace();
			return false;
		} finally {
			if (os != null) {
				try {
					os.close();
				} catch (IOException e) {
				}
			}
			if (fs != null) {
				try {
					fs.close();
				} catch (IOException e) {
				}
			}
		}
	}

	public synchronized boolean deserialize(String filename) {
		FileInputStream fs = null;
		ObjectInputStream is = null;
		try {
			fs = new FileInputStream(filename);
			is = new ObjectInputStream(fs);
			Territorium ter = (Territorium) is.readObject();
			int row = is.readInt();
			int col = is.readInt();
			int dir = is.readInt();
			int grains = is.readInt();
			is.close();
			this.rows = ter.rows;
			this.cols = ter.cols;
			this.tiles = ter.tiles;
			
			CompileManager.getCompileManager().loadAndSetNewHamster();
			// notwendig wenn Attribute im Hamster-Programm verwendet werden
			
			this.hamster.hamsterRow = row;
			this.hamster.hamsterCol = col;
			this.hamster.hamsterDirection = dir;
			this.hamster.hamsterGrains = grains;
			this.notifyObservers();
			return true;
		} catch (ClassNotFoundException e) {
			System.err.println(e.toString());
		} catch (IOException e) {
			System.err.println(e.toString());
		} catch (Throwable e) {
			System.err.println(e.toString());
		} finally {
			if (is != null) {
				try {
					is.close();
				} catch (IOException e) {
				}
			}
			if (fs != null) {
				try {
					fs.close();
				} catch (IOException e) {
				}
			}
		}
		return false;
	}

	public synchronized boolean save(String filename) {
		try {
			File xmlFile = new File(filename);

			XMLOutputFactory factory = XMLOutputFactory.newInstance();
			XMLStreamWriter writer = factory.createXMLStreamWriter(
					new FileOutputStream(xmlFile.getAbsolutePath()), "utf-8");

			// Der XML-Header wird erzeugt
			writer.writeStartDocument("utf-8", "1.0");
			writer.writeCharacters("\n");
			writer.writeDTD("<!DOCTYPE territorium [\n"

					+ "<!ELEMENT territorium ((wall | grains)*, hamster, (wall | grains)*)>\n"
					+ "<!ATTLIST territorium\n" + "    rows CDATA #REQUIRED\n"
					+ "    cols CDATA #REQUIRED\n" + ">\n"

					+ "<!ELEMENT wall EMPTY>\n" + "<!ATTLIST wall\n"
					+ "    row CDATA #REQUIRED\n" + "    col CDATA #REQUIRED\n"
					+ ">\n"

					+ "<!ELEMENT grains (#PCDATA)>\n" + "<!ATTLIST grains\n"
					+ "    row CDATA #REQUIRED\n" + "    col CDATA #REQUIRED\n"
					+ ">\n"

					+ "<!ELEMENT hamster EMPTY>\n" + "<!ATTLIST hamster\n"
					+ "    row CDATA #REQUIRED\n" + "    col CDATA #REQUIRED\n"
					+ "    direction CDATA #REQUIRED\n"
					+ "    grains CDATA #REQUIRED\n" + ">\n"

					+ "]>");
			writer.writeCharacters("\n");

			writer.writeStartElement("territorium");
			writer.writeAttribute("rows", "" + this.rows);
			writer.writeAttribute("cols", "" + this.cols);
			writer.writeCharacters("\n");

			for (int r = 0; r < this.rows; r++) {
				for (int c = 0; c < this.cols; c++) {
					if (this.tiles[r][c] == Territorium.WALL_ON_TILE) {
						writer.writeStartElement("wall");
						writer.writeAttribute("row", "" + r);
						writer.writeAttribute("col", "" + c);
						writer.writeEndElement(); // wall
						writer.writeCharacters("\n");
					} else if (this.tiles[r][c] > 0) {
						writer.writeStartElement("grains");
						writer.writeAttribute("row", "" + r);
						writer.writeAttribute("col", "" + c);
						writer.writeCharacters("" + this.tiles[r][c]);
						writer.writeEndElement(); // wall
						writer.writeCharacters("\n");
					}
				}
			}

			writer.writeStartElement("hamster");
			writer.writeAttribute("row", "" + this.hamster.hamsterRow);
			writer.writeAttribute("col", "" + this.hamster.hamsterCol);
			writer.writeAttribute("direction", ""
					+ this.hamster.hamsterDirection);
			writer.writeAttribute("grains", "" + this.hamster.hamsterGrains);
			writer.writeEndElement(); // hamster
			writer.writeCharacters("\n");

			writer.writeEndElement(); // territorium
			writer.writeCharacters("\n");
			writer.writeEndDocument();
			writer.writeCharacters("\n");
			writer.close();

			return true;

		} catch (Throwable exc) {
			exc.printStackTrace();
		}
		return false;
	}

	private int rowI;
	private int colI;
	private boolean isGrainElement;

	public synchronized boolean loadStAXCursor(String filename) {
		this.rowI = 0;
		this.colI = 0;
		this.isGrainElement = false;
		this.clear();
		try {
			XMLInputFactory factory = XMLInputFactory.newInstance();
			XMLStreamReader parser = factory
					.createXMLStreamReader(new FileInputStream(filename));
			while (parser.hasNext()) {
				switch (parser.getEventType()) {
				case XMLStreamConstants.START_DOCUMENT:
					break;
				case XMLStreamConstants.END_DOCUMENT:
					parser.close();
					break;
				case XMLStreamConstants.START_ELEMENT:
					String element = parser.getLocalName();
					if ("territorium".equals(element)) {
						try {
							int newRows = Integer.parseInt(parser
									.getAttributeValue(null, "rows"));
							int newCols = Integer.parseInt(parser
									.getAttributeValue(null, "cols"));
							this.changeSize(newRows, newCols);
						} catch (Exception exc) {
							// fehlerhafte XML-Daten; Satz wird ignoriert
						}
					} else if ("wall".equals(element)) {
						try {
							int row = Integer.parseInt(parser
									.getAttributeValue(null, "row"));
							int col = Integer.parseInt(parser
									.getAttributeValue(null, "col"));
							this.tiles[row][col] = Territorium.WALL_ON_TILE;
						} catch (Exception exc) {
							// fehlerhafte XML-Daten; Satz wird ignoriert
						}
					} else if ("grains".equals(element)) {
						try {
							this.isGrainElement = true;
							this.rowI = Integer.parseInt(parser
									.getAttributeValue(null, "row"));
							this.colI = Integer.parseInt(parser
									.getAttributeValue(null, "col"));
						} catch (Exception exc) {
							this.isGrainElement = false;
							// fehlerhafte XML-Daten; Satz wird ignoriert
						}
					} else if ("hamster".equals(element)) {
						try {
							int row = Integer.parseInt(parser
									.getAttributeValue(null, "row"));
							int col = Integer.parseInt(parser
									.getAttributeValue(null, "col"));
							int direction = Integer.parseInt(parser
									.getAttributeValue(null, "direction"));
							int grains = Integer.parseInt(parser
									.getAttributeValue(null, "grains"));
							CompileManager.getCompileManager()
									.loadAndSetNewHamster();
							this.hamster.hamsterRow = row;
							this.hamster.hamsterCol = col;
							this.hamster.hamsterDirection = direction;
							this.hamster.hamsterGrains = grains;
						} catch (Exception exc) {
							// fehlerhafte XML-Daten; Satz wird ignoriert
						}
					}
					break;
				case XMLStreamConstants.CHARACTERS:
					if (this.isGrainElement) {
						try {
							if (!parser.isWhiteSpace()) {
								System.out.println(parser.getText());
								int number = Integer.parseInt(parser.getText()
										.trim());
								this.tiles[this.rowI][this.colI] = number;
							}
						} catch (Exception exc) {
							exc.printStackTrace();
							// fehlerhafte XML-Daten; Satz wird ignoriert
						} finally {
							this.isGrainElement = false;
						}
					}
					break;
				default:
					break;
				}
				parser.next();
			}
			return true;
		} catch (Exception exc) {
			return false;
		} finally {
			this.notifyObservers();
		}
	}

	public synchronized boolean loadStAXIterator(String filename) {
		this.rowI = 0;
		this.colI = 0;
		this.isGrainElement = false;
		this.clear();
		try {
			InputStream in = new FileInputStream(filename);
			XMLInputFactory factory = XMLInputFactory.newInstance();
			XMLEventReader parser = factory.createXMLEventReader(in);
			while (parser.hasNext()) {
				XMLEvent event = parser.nextEvent();
				switch (event.getEventType()) {
				case XMLStreamConstants.START_DOCUMENT:
					break;
				case XMLStreamConstants.END_DOCUMENT:
					parser.close();
					break;
				case XMLStreamConstants.START_ELEMENT:
					StartElement element = event.asStartElement();
					if ("territorium".equals(element.getName().getLocalPart())) {
						try {
							int newRows = Integer.parseInt(element
									.getAttributeByName(new QName("rows"))
									.getValue());
							int newCols = Integer.parseInt(element
									.getAttributeByName(new QName("cols"))
									.getValue());
							this.changeSize(newRows, newCols);
						} catch (Exception exc) {
							// fehlerhafte XML-Daten; Satz wird ignoriert
						}
					} else if ("wall".equals(element.getName().getLocalPart())) {
						try {
							int row = Integer.parseInt(element
									.getAttributeByName(new QName("row"))
									.getValue());
							int col = Integer.parseInt(element
									.getAttributeByName(new QName("col"))
									.getValue());
							this.tiles[row][col] = Territorium.WALL_ON_TILE;
						} catch (Exception exc) {
							// fehlerhafte XML-Daten; Satz wird ignoriert
						}
					} else if ("grains"
							.equals(element.getName().getLocalPart())) {
						this.isGrainElement = true;
						try {
							this.rowI = Integer.parseInt(element
									.getAttributeByName(new QName("row"))
									.getValue());
							this.colI = Integer.parseInt(element
									.getAttributeByName(new QName("col"))
									.getValue());
						} catch (Exception exc) {
							this.isGrainElement = false;
							// fehlerhafte XML-Daten; Satz wird ignoriert
						}
					} else if ("hamster".equals(element.getName()
							.getLocalPart())) {
						try {
							int row = Integer.parseInt(element
									.getAttributeByName(new QName("row"))
									.getValue());
							int col = Integer.parseInt(element
									.getAttributeByName(new QName("col"))
									.getValue());
							int direction = Integer.parseInt(element
									.getAttributeByName(new QName("direction"))
									.getValue());
							int grains = Integer.parseInt(element
									.getAttributeByName(new QName("grains"))
									.getValue());
							CompileManager.getCompileManager()
									.loadAndSetNewHamster();
							this.hamster.hamsterRow = row;
							this.hamster.hamsterCol = col;
							this.hamster.hamsterDirection = direction;
							this.hamster.hamsterGrains = grains;
						} catch (Exception exc) {
							// fehlerhafte XML-Daten; Satz wird ignoriert
						}
					}
					break;
				case XMLStreamConstants.CHARACTERS:
					if (this.isGrainElement) {
						try {
							Characters chars = event.asCharacters();
							if (!chars.isWhiteSpace()) {
								int number = Integer.parseInt(chars.toString()
										.trim());
								this.tiles[this.rowI][this.colI] = number;
							}
						} catch (Exception exc) {
							// fehlerhafte XML-Daten; Satz wird ignoriert
						} finally {
							this.isGrainElement = false;
						}
					}
					break;
				default:
					break;
				}
			}
			return true;
		} catch (Exception exc) {
			return false;
		} finally {
			this.notifyObservers();
		}

	}

	public synchronized boolean loadDOM(String filename) {
		this.clear();
		try {
			DocumentBuilderFactory factory = DocumentBuilderFactory
					.newInstance();
			factory.setValidating(true);
			factory.setIgnoringComments(true);
			DocumentBuilder builder = factory.newDocumentBuilder();
			builder.setErrorHandler(null);
			Document document = builder.parse(filename);

			NodeList l = document.getElementsByTagName("territorium");
			if (l.getLength() > 0) {
				try {
					Element e = (Element) l.item(0);
					int newRows = Integer.parseInt(e.getAttribute("rows"));
					int newCols = Integer.parseInt(e.getAttribute("cols"));
					this.changeSize(newRows, newCols);
				} catch (Exception exc) {
					// fehlerhafte XML-Daten; Satz wird ignoriert
				}
			}

			l = document.getElementsByTagName("wall");
			for (int i = 0; i < l.getLength(); i++) {
				try {
					Element e = (Element) l.item(i);
					int row = Integer.parseInt(e.getAttribute("row"));
					int col = Integer.parseInt(e.getAttribute("col"));
					this.tiles[row][col] = Territorium.WALL_ON_TILE;
				} catch (Exception exc) {
					// fehlerhafte XML-Daten; Satz wird ignoriert
				}
			}

			l = document.getElementsByTagName("grains");
			for (int i = 0; i < l.getLength(); i++) {
				try {
					Element e = (Element) l.item(i);
					int row = Integer.parseInt(e.getAttribute("row"));
					int col = Integer.parseInt(e.getAttribute("col"));
					NodeList l2 = e.getChildNodes();
					if (l2.getLength() > 0) {
						int number = Integer
								.parseInt(l2.item(0).getNodeValue());
						this.tiles[row][col] = number;
					}
				} catch (Exception exc) {
					// fehlerhafte XML-Daten; Satz wird ignoriert
				}
			}

			l = document.getElementsByTagName("hamster");
			if (l.getLength() > 0) {
				try {
					Element e = (Element) l.item(0);
					int row = Integer.parseInt(e.getAttribute("row"));
					int col = Integer.parseInt(e.getAttribute("col"));
					int direction = Integer.parseInt(e
							.getAttribute("direction"));
					int grains = Integer.parseInt(e.getAttribute("grains"));
					CompileManager.getCompileManager().loadAndSetNewHamster();
					this.hamster.hamsterRow = row;
					this.hamster.hamsterCol = col;
					this.hamster.hamsterDirection = direction;
					this.hamster.hamsterGrains = grains;
				} catch (Exception exc) {
					// fehlerhafte XML-Daten; Satz wird ignoriert
				}
			}
			return true;
		} catch (Exception exc) {
			return false;
		} finally {
			this.notifyObservers();
		}
	}

	public synchronized boolean loadSAX(String filename) {
		this.clear();
		try {
			// Use an instance of ourselves as the SAX event handler
			DefaultHandler handler = new SAXHandler(this);
			// Parse the input with the default (non-validating) parser
			SAXParserFactory factory = SAXParserFactory.newInstance();
			factory.setValidating(true);
			SAXParser saxParser = factory.newSAXParser();
			saxParser.parse(new File(filename), handler);
			return true;
		} catch (Throwable exc) {
			return false;
		} finally {
			this.notifyObservers();
		}
	}
	
	

	@Override
	public int getNumberOfRows() {
		
		return this.rows;
	}

	@Override
	public int getNumberOfColumns() {
		
		return this.cols;
	}

	@Override
	public int getNumberOfGrainsOnTile(int row, int column) {
		
		return this.getNoOfGrains(row, column);
	}

	@Override
	public IHamster getHamsterState() {
		
		return this.hamster;
	}
	
	public void setHamsterDirection(int direction) {
		
		this.hamster.setDirection(direction);
	}
	
	public void setHamsterNumberOfGrains(int numberOfGrains) {
		
		this.hamster.setNumberOfGrains(numberOfGrains);
	}
}

class SAXHandler extends DefaultHandler {

	private Territorium territorium;
	private boolean isGrainElement;
	private int row;
	private int col;
	private int number;

	public SAXHandler(Territorium territorium) {
		this.territorium = territorium;
		this.isGrainElement = false;
	}

	@Override
	public void startDocument() throws SAXException {
	}

	@Override
	public void endDocument() throws SAXException {
	}

	@Override
	public void startElement(String namespaceURI, String localName,
			String qName, Attributes attrs) throws SAXException {
		if ("territorium".equals(qName)) {
			try {
				int newRows = Integer.parseInt(attrs.getValue("rows"));
				int newCols = Integer.parseInt(attrs.getValue("cols"));
				this.territorium.changeSize(newRows, newCols);
			} catch (Exception exc) {
				// fehlerhafte XML-Daten; Satz wird ignoriert
			}
		} else if ("wall".equals(qName)) {
			try {
				int row = Integer.parseInt(attrs.getValue("row"));
				int col = Integer.parseInt(attrs.getValue("col"));
				this.territorium.tiles[row][col] = Territorium.WALL_ON_TILE;
			} catch (Exception exc) {
				// fehlerhafte XML-Daten; Satz wird ignoriert
			}
		} else if ("grains".equals(qName)) {
			this.isGrainElement = true;
			try {
				this.row = Integer.parseInt(attrs.getValue("row"));
				this.col = Integer.parseInt(attrs.getValue("col"));
			} catch (Exception exc) {
				this.isGrainElement = false;
				// fehlerhafte XML-Daten; Satz wird ignoriert
			}
		} else if ("hamster".equals(qName)) {
			try {
				int row = Integer.parseInt(attrs.getValue("row"));
				int col = Integer.parseInt(attrs.getValue("col"));
				int direction = Integer.parseInt(attrs.getValue("direction"));
				int grains = Integer.parseInt(attrs.getValue("grains"));
				CompileManager.getCompileManager().loadAndSetNewHamster();
				Hamster newHamster = this.territorium.getHamster();
				newHamster.hamsterRow = row;
				newHamster.hamsterCol = col;
				newHamster.hamsterDirection = direction;
				newHamster.hamsterGrains = grains;
			} catch (Exception exc) {
				// fehlerhafte XML-Daten; Satz wird ignoriert
			}
		}
	}

	@Override
	public void endElement(String namespaceURI, String localName, String qName)
			throws SAXException {
		if (this.isGrainElement) {
			try {
				this.territorium.tiles[this.row][this.col] = this.number;
			} catch (Exception exc) {
				// fehlerhafte XML-Daten; Satz wird ignoriert
			} finally {
				this.isGrainElement = false;
			}
		}
	}

	@Override
	public void characters(char[] buf, int offset, int len) throws SAXException {
		if (this.isGrainElement) {
			try {
				String numberStr = new String(buf, offset, len);
				this.number = Integer.parseInt(numberStr.trim());
				this.territorium.tiles[this.row][this.col] = number;
			} catch (Exception exc) {
				// fehlerhafte XML-Daten; Satz wird ignoriert
			} finally {
				this.isGrainElement = false;
			}
		}
	}
}
