package view;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Image;
import java.awt.image.BufferedImage;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.io.File;

import javax.imageio.ImageIO;
import javax.swing.BorderFactory;
import javax.swing.Icon;
import javax.swing.ImageIcon;
import javax.swing.JFileChooser;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.SwingConstants;
import javax.swing.filechooser.FileNameExtensionFilter;
import javax.swing.filechooser.FileView;

import util.ResourceLoader;
import util.ResourceManager;

/**
 * Chooser zum Auswhlen eines Images
 * 
 * @author Dietrich Boles, Uni Oldenburg
 * @version 1.0 (12.11.2008)
 * 
 */
public class JImageChooser extends JFileChooser {

	private static JImageChooser chooser = null;
	private static JImageChooser playChooser = null;

	public static JImageChooser getInstance() {
		if (chooser == null) {
			chooser = new JImageChooser();
		}
		return chooser;
	}

	public static JImageChooser getPlayInstance(String path) {
		if (playChooser == null) {
			playChooser = new JImageChooser(path);
			playChooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
		}
		return playChooser;
	}

	private PreviewPane previewPane;

	private JImageChooser() {
		this(null);
	}

	private JImageChooser(String path) {
		super(path);
		this.previewPane = new PreviewPane();
		this.setAcceptAllFileFilterUsed(false);
		this.setAccessory(this.previewPane);
		this.addPropertyChangeListener(this.previewPane);
		FileNameExtensionFilter filter = new FileNameExtensionFilter(
				ResourceManager.getResourceManager().getValue(
						"imagechooser.open.typedesc"), "jpg", "gif", "png",
				"bmp");
		this.setFileFilter(filter);
		this.setFileView(new ImageFileView());
	}
}

class PreviewPane extends JPanel implements PropertyChangeListener {
	private JLabel imgLabel;
	private JLabel infoLabel;
	private int maxImgWidth;

	public PreviewPane() {
		this.setLayout(new BorderLayout(5, 5));
		this.setBorder(BorderFactory.createEmptyBorder(5, 5, 5, 5));
		this.add(new JLabel("Preview:"), BorderLayout.NORTH);
		this.imgLabel = new JLabel();
		this.imgLabel.setBackground(Color.WHITE);
		this.imgLabel.setOpaque(true);
		this.imgLabel.setHorizontalAlignment(SwingConstants.CENTER);
		this.imgLabel.setPreferredSize(new Dimension(200, 200));
		this.maxImgWidth = 195;
		this.imgLabel.setBorder(BorderFactory.createEtchedBorder());
		this.add(this.imgLabel, BorderLayout.CENTER);
		this.infoLabel = new JLabel();
		this.add(this.infoLabel, BorderLayout.SOUTH);
	}

	public void propertyChange(PropertyChangeEvent evt) {
		Icon icon = null;
		String infoText = "";
		if (JFileChooser.SELECTED_FILE_CHANGED_PROPERTY.equals(evt
				.getPropertyName())) {
			File newFile = (File) evt.getNewValue();
			if (newFile != null) {
				String path = newFile.getAbsolutePath();
				if (path.toLowerCase().endsWith(".gif")
						|| path.toLowerCase().endsWith(".jpg")
						|| path.toLowerCase().endsWith(".png")
						|| path.toLowerCase().endsWith(".bmp")) {
					try {
						BufferedImage img = ImageIO.read(newFile);
						float width = img.getWidth();
						float height = img.getHeight();
						if (width > this.maxImgWidth
								|| height > this.maxImgWidth) {
							float scale = height / width;
							width = this.maxImgWidth;
							height = width * scale; // height should be
							// scaled from new width
							icon = new ImageIcon(img.getScaledInstance(Math
									.max(1, (int) width), Math.max(1,
									(int) height), Image.SCALE_SMOOTH));
						} else {
							icon = new ImageIcon(img);
						}
						infoText += img.getWidth() + " x " + img.getHeight();
						infoText += ", " + newFile.length() / 1024 + " KB";
					} catch (Throwable e) {
						e.printStackTrace();
					}
				}
			}

			this.imgLabel.setIcon(icon);
			this.infoLabel.setText(infoText);

			this.repaint();

		}
	}

}

class ImageFileView extends FileView {

	public Icon getIcon(File f) {
		String path = f.getAbsolutePath();
		if (path.toLowerCase().endsWith(".gif")
				|| path.toLowerCase().endsWith(".jpg")
				|| path.toLowerCase().endsWith(".png")
				|| path.toLowerCase().endsWith(".bmp")) {
			return ResourceLoader.getImageIcon("image.png");
		}
		return super.getIcon(f);
	}
}
