package theater_intern;

import java.awt.Rectangle;
import java.io.Serializable;
import java.util.HashMap;

import model.Play;
import theater.Component;
import theater.PixelArea;
import theater.TheaterImage;

/**
 * Interne Umsetzung einer Componente
 * 
 * @author Dietrich Boles, Uni Oldenburg
 * @version 1.0 (12.11.2008)
 * 
 */
public class IComponent implements Serializable {

	/***************************************************************************
	 * * * * * * * * * * * * * * Umformungsmethoden
	 */
	private static HashMap<Component, IComponent> compMap = new HashMap<Component, IComponent>();
	private static HashMap<IComponent, Component> iCompMap = new HashMap<IComponent, Component>();

	public static void newCompIComp(Component comp, IComponent iComp) {
		IComponent.compMap.put(comp, iComp);
		IComponent.iCompMap.put(iComp, comp);
	}

	public static Component getComp(IComponent iComp) {
		return IComponent.iCompMap.get(iComp);
	}

	public static IComponent getIComp(Component comp) {
		return IComponent.compMap.get(comp);
	}

	/***************************************************************************
	 * * * * * * * * * * * * * * Componentmethoden
	 */

	transient private IStage stage;
	
	private TheaterImage image;
	private int rotation;

	private int column;
	private int row;
	private int zCoordinate;
	private boolean visible;

	private boolean handlingKeyEvents;
	private boolean handlingMouseEvents;

	public IComponent(boolean isActor) {
		this.stage = null;
		this.column = -1;
		this.row = -1;
		this.zCoordinate = Component.DEF_Z_COORDINATE;
		this.visible = true;
		this.rotation = 0;
		this.image = null;
		this.handlingKeyEvents = false;
		this.handlingMouseEvents = false;
	}
	
	public IComponent(IComponent old) {
		this.stage = null;
		this.column = 0;
		this.row = 0;
		this.zCoordinate = old.zCoordinate;
		this.visible = old.visible;
		this.rotation = old.rotation;
		this.image = old.image;
		this.handlingKeyEvents = old.handlingKeyEvents;
		this.handlingMouseEvents = old.handlingMouseEvents;
	}
	
	public void copyAttributes(IComponent old) {
		this.stage = null;
		this.column = old.column;
		this.row = old.row;
		this.zCoordinate = old.zCoordinate;
		this.visible = old.visible;
		this.rotation = old.rotation;
		this.image = old.image;
		this.handlingKeyEvents = old.handlingKeyEvents;
		this.handlingMouseEvents = old.handlingMouseEvents;
	}

	public void setImage(TheaterImage image) {
		if (image != this.image) {
			this.image = image;
			TheaterObservable.getObservable().stateChange();
		}
	}

	public void setImage(String filename) throws IllegalArgumentException {
		this.setImage(new TheaterImage(filename));
	}

	public TheaterImage getImage() {
		return this.image;
	}

	public IStage getStage() {
		return this.stage;
	}

	public int getColumn() {
		this.failIfNotInStage();
		return this.column;
	}

	public int getRow() {
		this.failIfNotInStage();
		return this.row;
	}

	public int getZCoordinate() {
		// this.failIfNotInStage();
		return this.zCoordinate;
	}

	public int getRotation() {
		return this.rotation;
	}

	public boolean isVisible() {
		return this.visible;
	}

	public void setLocation(int newCol, int newRow) {
		this.failIfNotInStage();
		if (this.column != newCol || this.row != newRow) {
			this.column = newCol;
			this.row = newRow;
			TheaterObservable.getObservable().stateChange();
		}
	}

	public void setZCoordinate(int newZ) {
		if (this.zCoordinate != newZ) {
			this.zCoordinate = newZ;
			TheaterObservable.getObservable().stateChange();
		}
	}

	public void setRotation(int rotation) {
		if (this.rotation != rotation) {
			this.rotation = rotation;
			TheaterObservable.getObservable().stateChange();
		}
	}

	public void setVisible(boolean visible) {
		if (this.visible != visible) {
			this.visible = visible;
			TheaterObservable.getObservable().stateChange();
		}
	}

	public Rectangle getImageRectangle() {
		this.failIfNotInStage();
		return Play.getPlay().getStagePanel().getStageRectangle(this);
	}

	void setStage(IStage stage) {
		this.stage = stage;
	}

	void setColumn(int newCol) {
		this.column = newCol;
	}

	void setRow(int newRow) {
		this.row = newRow;
	}

	void setZ(int newZ) {
		this.zCoordinate = newZ;
	}

	private void failIfNotInStage() {
		if (this.stage == null) {
			throw new IllegalStateException();
		}
	}

	public boolean contains(int x, int y) {
		this.failIfNotInStage();
		java.awt.Rectangle imgRect = Play.getPlay().getStagePanel()
				.getStageRectangle(this);
		return new theater.Rectangle(imgRect).contains(x, y);
	}

	public boolean isInside(PixelArea area) {
		this.failIfNotInStage();
		java.awt.Rectangle imgRect = Play.getPlay().getStagePanel()
				.getStageRectangle(this);
		return new theater.Rectangle(imgRect).isInside(area);
	}

	public boolean intersects(PixelArea area) {
		this.failIfNotInStage();
		java.awt.Rectangle imgRect = Play.getPlay().getStagePanel()
				.getStageRectangle(this);
		return new theater.Rectangle(imgRect).intersects(area);
	}

	public boolean isHandlingKeyEvents() {
		return this.handlingKeyEvents;
	}

	public void setHandlingKeyEvents(boolean handlingKeyEvents) {
		this.handlingKeyEvents = handlingKeyEvents;
	}

	public boolean isHandlingMouseEvents() {
		return this.handlingMouseEvents;
	}

	public void setHandlingMouseEvents(boolean handlingMouseEvents) {
		this.handlingMouseEvents = handlingMouseEvents;
	}
	
}
