package scratch.gui;

import java.awt.Color;
import java.awt.Font;
import java.awt.Graphics;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.image.BufferedImage;
import java.util.ArrayList;

import scratch.ScratchUtils;

public class TabButton implements MouseListener {
	public static Color DARK_TEXT_COLOR = new Color(33, 33, 31);
	public static Color LIGHT_TEXT_COLOR = new Color(88, 89, 93);
	public static final Font TEXT_FONT = new Font("Verdana", Font.BOLD, 11);
	public static final int PADDING = 15;
	public static final int SPACING = -6;
	public static final int HEIGHT = 23;

	private static BufferedImage imgLeft;
	private static BufferedImage imgRight;
	private static BufferedImage imgMiddle;
	private static BufferedImage imgLeftHovered;
	private static BufferedImage imgRightHovered;
	private static BufferedImage imgMiddleHovered;

	private ArrayList<RefreshHandler> refreshHandler;
	private ArrayList<TabChangedHandler> tabChangedHandler;
	private ArrayList<TabClickHandler> clickHandler;
	private String name;
	private int textWidth;
	private int textHeight;
	private int x;
	private int y;
	private boolean selected;
	private boolean hovered;
	private boolean visible;

	public TabButton(String name) {
		this.name = name;
		this.textWidth = ScratchUtils.getTextWidth(name, TabButton.TEXT_FONT);
		this.textHeight = ScratchUtils.getTextHeight("Ag", TabButton.TEXT_FONT);
		this.refreshHandler = new ArrayList<RefreshHandler>();
		this.tabChangedHandler = new ArrayList<TabChangedHandler>();
		this.clickHandler = new ArrayList<TabClickHandler>();
		this.loadImages();
	}

	/**
	 * Wird im Konstruktor aufgerufen und ldt LookAndFeel Bilder. Diese Methode
	 * wird nur einmalig ausgefhrt, da die Bilder statisch in der Klasse sind.
	 */
	private void loadImages() {
		if (TabButton.imgLeft != null) {
			return; // Bilder sind bereits geladen
		}

		TabButton.imgLeft = ScratchUtils.getImage("scratch/buttons/tab_l.png");
		TabButton.imgRight = ScratchUtils.getImage("scratch/buttons/tab_r.png");
		TabButton.imgMiddle = ScratchUtils
				.getImage("scratch/buttons/tab_m.png");
		TabButton.imgLeftHovered = ScratchUtils
				.getImage("scratch/buttons/tab_l_hover.png");
		TabButton.imgRightHovered = ScratchUtils
				.getImage("scratch/buttons/tab_r_hover.png");
		TabButton.imgMiddleHovered = ScratchUtils
				.getImage("scratch/buttons/tab_m_hover.png");
	}

	public void paint(Graphics g, int x, int y) {
		this.x = x;
		this.y = y;
		this.paintBackground(g, x, y);

		g.setColor(this.getTextColor());
		g.setFont(TabButton.TEXT_FONT);
		g.drawString(this.name, x + TabButton.PADDING, y
				+ (TabButton.HEIGHT + this.textHeight) / 2 - 3);
	}

	private void paintBackground(Graphics g, int x, int y) {
		g.drawImage(this.getLeftImage(), x, y, null);
		g.drawImage(this.getRightImage(), x + TabButton.PADDING
				+ this.textWidth, y, null);

		// g.drawImage(getMiddleImage(), x + PADDING, y, textWidth, HEIGHT,
		// null);
		for (int i = 0; i < this.getMiddleImage().getHeight(); i++) {
			g.setColor(new Color(this.getMiddleImage().getRGB(0, i)));
			g.drawLine(x + TabButton.PADDING, y + i, x + TabButton.PADDING
					+ this.textWidth, y + i);
		}
	}

	private BufferedImage getLeftImage() {
		if (this.selected || this.hovered) {
			return TabButton.imgLeftHovered;
		}

		return TabButton.imgLeft;
	}

	private BufferedImage getRightImage() {
		if (this.selected || this.hovered) {
			return TabButton.imgRightHovered;
		}

		return TabButton.imgRight;
	}

	private BufferedImage getMiddleImage() {
		if (this.selected || this.hovered) {
			return TabButton.imgMiddleHovered;
		}

		return TabButton.imgMiddle;
	}

	public void setSelected(boolean value) {
		this.selected = value;
	}

	public boolean isSelected() {
		return this.selected;
	}

	public void setVisible(boolean value) {
		this.visible = value;
	}

	public boolean isVisible() {
		return this.visible;
	}

	public String getName() {
		return this.name;
	}

	public void setName(String name) {
		this.name = name;
		this.textWidth = ScratchUtils.getTextWidth(name, TabButton.TEXT_FONT);
	}

	public int getWidth() {
		return this.textWidth + 2 * TabButton.PADDING + TabButton.SPACING;
	}

	private boolean hitTest(int x, int y) {
		if (x < this.x || x > this.x + this.getWidth()) {
			return false;
		}

		if (y < this.y || y > this.y + TabButton.HEIGHT) {
			return false;
		}

		return true;
	}

	/**
	 * Fhrt fr alle angemeldeten TabClickHandler die darin enthaltene Methode
	 * onClick(TabButton) aus und bergibt dabei diesen TabButton als Sender und
	 * die angeklickte Koordinate.
	 * 
	 * @param x
	 *            x Koordinate
	 * @param y
	 *            y Koordinate
	 */
	public void onRightClick(int x, int y) {
		for (TabClickHandler handler : this.clickHandler) {
			handler.onClick(this, x, y);
		}
	}

	/**
	 * Fgt diesem TabButton einen TabClickHandler hinzu
	 * 
	 * @param handler
	 */
	public void addTabClickHandler(TabClickHandler handler) {
		this.clickHandler.add(handler);
	}

	public void addRefreshHandler(RefreshHandler handler) {
		this.refreshHandler.add(handler);
	}

	/**
	 * Fgt einen TabChangedHandler zu diesem TabPanel hinzu, dessen interne
	 * Methode tabChanged aufgerufen wird, sobald sich der aktive Tab ndert.
	 * 
	 * @param handler
	 *            Handler, der beim Tabwechsel aufgerufen werden soll
	 */
	public void addTabChangedHandler(TabChangedHandler handler) {
		this.tabChangedHandler.add(handler);
	}

	private void refresh() {
		for (RefreshHandler handler : this.refreshHandler) {
			handler.refresh();
		}
	}

	private boolean tabChanged() {
		if (this.selected) {
			return false;
		}

		for (TabChangedHandler handler : this.tabChangedHandler) {
			handler.tabChanged(this);
		}
		return true;
	}

	private Color getTextColor() {
		if (this.selected) {
			return TabButton.DARK_TEXT_COLOR;
		}

		return TabButton.LIGHT_TEXT_COLOR;
	}

	@Override
	public void mouseClicked(MouseEvent arg0) {
		// TODO Auto-generated method stub

	}

	@Override
	public void mouseEntered(MouseEvent arg0) {
		// TODO Auto-generated method stub

	}

	@Override
	public void mouseExited(MouseEvent arg0) {
		// TODO Auto-generated method stub

	}

	@Override
	public void mousePressed(MouseEvent e) {
		if (!this.visible) {
			return;
		}

		if (this.hitTest(e.getX(), e.getY())) {
			this.hovered = true;
			this.refresh();
		}
	}

	@Override
	public void mouseReleased(MouseEvent e) {
		if (!this.visible) {
			return;
		}

		if (this.hovered) {
			this.hovered = false;
			if (this.hitTest(e.getX(), e.getY())) {
				if (e.getButton() == MouseEvent.BUTTON1) {
					this.tabChanged();
				} else if (e.getButton() == MouseEvent.BUTTON3) {
					this.onRightClick(e.getX(), e.getY());
				}
			}

			this.refresh();
		}
	}
}
