package scratch;

import java.awt.EventQueue;
import java.io.File;
import java.util.ArrayList;
import java.util.Stack;

import javax.swing.JOptionPane;
import javax.swing.SwingUtilities;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import model.Play;

import org.w3c.dom.Document;
import org.w3c.dom.NodeList;

import scratch.elements.voids.ExitProgramException;
import scratch.elements.voids.FunctionResultException;
import scratch.gui.NextMethodHandler;
import scratch.gui.RefreshHandler;
import simulation.SimulationManager;
import theater_intern.StopException;
import debugger.SolistDebugger;

/**
 * Ein ScratchProgramm beinhaltet eine Sequenz von Anweisungen (hier Statements)
 * 
 * @author dibo
 * 
 */
public class ScratchProgram extends Thread {
	private static Renderable active;

	private boolean setStopped;
	private boolean setPaused;
	private ScratchFile file;
	private StorageController program;
	private ArrayList<RefreshHandler> refreshHandler = new ArrayList<RefreshHandler>();
	private ArrayList<NextMethodHandler> nextMethodHandler = new ArrayList<NextMethodHandler>();
	private String openedMethod = "";
	private Stack<String> methodStack = new Stack<String>();
	private boolean thrownException;

	public ScratchProgram() {
		super();
		this.setStopped = false;
		this.setPaused = false;
		this.program = new StorageController();
	}

	public ScratchProgram(ScratchProgram p) {
		super(p);
		this.setStopped = p.setStopped;
		this.setPaused = p.setPaused;
		this.file = p.file;
		this.program = p.program;
		this.refreshHandler = p.refreshHandler;
		this.nextMethodHandler = p.nextMethodHandler;
		this.openedMethod = p.openedMethod;
		this.methodStack = p.methodStack;
	}

	public void setFile(ScratchFile f) {
		this.file = f;
	}

	public void setProgram(StorageController program) {
		this.program = program;
	}

	public StorageController getProgram() {
		return this.program;
	}

	public static void setActive(Renderable r) {
		ScratchProgram.active = r;
	}

	public static boolean isActive(Renderable r) {
		return ScratchProgram.active == r;
	}

	public ArrayList<RefreshHandler> getRefreshHandler() {
		return this.refreshHandler;
	}

	public void setRefreshHandler(ArrayList<RefreshHandler> refreshHandler) {
		this.refreshHandler = refreshHandler;
	}

	public void addRefreshHandler(RefreshHandler handler) {
		this.refreshHandler.add(handler);
	}

	public ArrayList<NextMethodHandler> getNextMethodHandler() {
		return this.nextMethodHandler;
	}

	public void setNextMethodHandler(
			ArrayList<NextMethodHandler> nextMethodHandler) {
		this.nextMethodHandler = nextMethodHandler;
	}

	public void addNextMethodHandler(NextMethodHandler handler) {
		this.nextMethodHandler.add(handler);
	}

	public void nexMethod(String name) {
		this.methodStack.push(name);
	}

	public void endMmethod() {
		this.methodStack.pop();
	}

	public void refresh() {
		for (RefreshHandler handler : this.refreshHandler) {
			handler.refresh();
		}
	}

	private void onMethodChanged() {
		for (NextMethodHandler handler : this.nextMethodHandler) {
			handler.nextMethod(this.methodStack.peek());
		}
	}

	@Override
	public void run() {
		this.methodStack = new Stack<String>();
		this.methodStack.push("main");
		this.openedMethod = "";
		thrownException = false;
		Renderable rootElement = this.program.getMainRoot();
		try {
			rootElement.perform(this);
		} catch (final ExitProgramException e) {
			if (!ThreadDeath.class.isInstance(e.getThrowable())) {
				thrownException = true;
				EventQueue.invokeLater(new Runnable() {
					public void run() {
						JOptionPane.showMessageDialog(Play.getPlay()
								.getStagePanel(), e.getThrowable().toString(),
								e.getThrowable().getClass().getName(),
								JOptionPane.ERROR_MESSAGE);
					}
				});
			}
		} catch (FunctionResultException e) {
			// Main Funktion wurde mit return verlassen,
			// es soll nichts weiter unternommen werden
		} catch (ThreadDeath th) {
		} catch (final Throwable exc) {
			SwingUtilities.invokeLater(new Runnable() {
				public void run() {
					JOptionPane.showMessageDialog(null, exc.toString(),
							"Scratch-Exception", JOptionPane.ERROR_MESSAGE,
							null);
				}
			});
		} finally {
			if (!thrownException) {
				ScratchProgram.active = null;
				this.refresh();
			}
			this.setStopped = false;
			this.setPaused = false;
			ScratchSolist.getScratchSolist().setProgramFinished();
			SwingUtilities.invokeLater(new Runnable() {
				public void run() {
					try {
						SimulationManager.getSimulationManager().handleEnd();
						Play.getPlay().getActivePerformance().stopSimulation();
					} catch (StopException exc) {

					}
				}
			});
		}
	}

	public synchronized void stopProgram() {
		this.notify(); // fr den Fall, dass das Programm gerade pausiert ist
		this.setStopped = true;
	}

	public synchronized void pauseProgram() {
		this.setPaused = true;
	}

	public synchronized void resumeProgram() {
		this.setPaused = false;
		ScratchProgram.active = null;
		this.refresh();
		this.notify();
	}

	public synchronized void stepOver() {
		this.notify();
		this.setPaused = true;
	}

	public synchronized void stepInto() {
		// in dieser Demo gibt es keine Prozeduren
		this.stepOver(); // Dummy
	}

	public synchronized boolean checkStop() {
		return this.setStopped;
	}

	public synchronized void checkPause() {
		if (this.setPaused) {
			// doTracing();
			try {
				this.wait();
			} catch (InterruptedException exc) {

			}
		}
	}

	public synchronized void doTracing() {
		if (Play.getPlay().getPlayFrame().getScratchPanel().isTracingOn()) {
			if (!this.openedMethod.equals(this.methodStack.peek())) {
				this.openedMethod = this.methodStack.peek();
				this.onMethodChanged();
			}
			this.refresh();
			SolistDebugger.sleepIntern();
		}
	}

	public void loadProgram(File xmlFile) throws Exception {
		this.setStopped = false;
		this.setPaused = false;
		this.program = new StorageController();

		DocumentBuilderFactory domFactory = DocumentBuilderFactory
				.newInstance();
		domFactory.setNamespaceAware(true);
		DocumentBuilder builder = domFactory.newDocumentBuilder();
		Document doc = builder.parse(xmlFile);
		NodeList methods = doc.getElementsByTagName("METHOD");
		this.program.loadProgram(methods);

	}
}
