package debugger;

import java.util.ArrayList;

import javax.swing.SwingUtilities;

import model.Play;
import simulation.SimulationManager;
import theater.Performance;
import editor.SolistEditor;

public class SolistDebugger {

	private static SolistDebugger debugger = new SolistDebugger();

	public static SolistDebugger getSolistDebugger() {
		return SolistDebugger.debugger;
	}

	private boolean on;
	private long currentLine;
	private ArrayList<Long> breakpoints;
	private long lastLine;
	private ArrayList<ArrayList<VariableInfo>> treeNodes;

	private ArrayList<FrameElement> frames;
	private ArrayList<VariableInfo> vars;

	protected SolistDebugger() {
		this.currentLine = -1;
		this.on = false;
		this.breakpoints = new ArrayList<Long>();
		this.lastLine = -2;
		this.treeNodes = new ArrayList<ArrayList<VariableInfo>>();
	}

	public void setTracing(boolean on) {
		this.on = on;
		if (this.on
				&& !Play.getPlay().getActivePerformance().simulationStopped()) {
			if (this.currentLine != -1) {
				SolistEditor.getSolistEditor().getEditorPanel().markLine(
						this.currentLine - 1);
			}
			if (this.frames != null) {
				DebuggerFrame.getDebuggerFrame().getStackFrameTable()
						.setStackFrames(this.frames);
			}
			if (this.vars != null) {
				DebuggerFrame.getDebuggerFrame().getVariablesTree()
						.setVariables(this.vars);
			}
		}
		if (!this.on) {
			this.clear();
		} else {
			DebuggerFrame frame = DebuggerFrame.getDebuggerFrame();
			if (!frame.isVisible()) {
				frame.toFront();
				frame.setVisible(true);
				Play.getPlay().getPlayFrame().getDebuggerFrameMenuItem()
						.setSelected(true);
				frame.toFront();
			}
		}
	}

	public boolean isTracingOn() {
		return this.on;
	}

	public void reset() {
		this.currentLine = -1;
		this.lastLine = -2;
		this.clear();
	}

	public void clear() {
		DebuggerFrame.getDebuggerFrame().getStackFrameTable().clear();
		DebuggerFrame.getDebuggerFrame().getVariablesTree().clear();
	}

	public void trace(int number, Object... variables) {
		Play.getPlay().getActivePerformance().checkStop();
		Performance.getPerformance().lock();
		try {
			Thread t = Thread.currentThread();
			if (SimulationManager.getSimulationManager().getSimulator() != t) {
				return;
			}

			StackTraceElement[] elems = t.getStackTrace();

			this.frames = this.buildStackFrame(elems);
			int stackSize = this.frames.size();

			this.vars = this.buildVariables(variables);

			if (stackSize <= this.treeNodes.size()) {
				this.treeNodes.set(stackSize - 1, this.vars);
			} else {
				this.treeNodes.add(this.vars);
			}

			if (this.isTracingOn()) {
				try {
					SwingUtilities.invokeAndWait(new Runnable() {
						public void run() {
							DebuggerFrame.getDebuggerFrame()
									.getStackFrameTable().setStackFrames(
											SolistDebugger.this.frames);
							DebuggerFrame.getDebuggerFrame().getVariablesTree()
									.setVariables(SolistDebugger.this.vars);
						}
					});
				} catch (Throwable th) {

				}
			} else if (SolistEditor.getSolistEditor().getEditorPanel()
					.lineHighlightSet()) {
				try {
					SwingUtilities.invokeAndWait(new Runnable() {
						public void run() {
							SolistEditor.getSolistEditor().getEditorPanel()
									.removeLineHighlight();
						}
					});
				} catch (Throwable th) {

				}
			}

			for (int i = 0; i < elems.length; i++) {
				StackTraceElement elem = elems[i];
				String fileName = elem.getFileName();
				if (fileName.equals(SolistEditor.CLASS_NAME + ".java")) {
					final int line = elem.getLineNumber();
					this.lastLine = this.currentLine;
					this.currentLine = line;
					if (this.isTracingOn()) {
						try {
							SwingUtilities.invokeAndWait(new Runnable() {
								public void run() {
									SolistEditor.getSolistEditor()
											.getEditorPanel()
											.markLine(line - 1);
								}
							});
						} catch (Throwable th) {

						}
						SolistDebugger.sleep();
					}
					this.checkBreakpoints();
					return;
				}
			}
		} finally {
			Performance.getPerformance().unlock();
			Play.getPlay().getActivePerformance().checkStep(number);
		}
	}

	public boolean traceB(int number, Object... variables) {
		this.trace(number, variables);
		return true;
	}

	public void setVariableTree(int index) {
		DebuggerFrame.getDebuggerFrame().getVariablesTree().setVariables(
				this.treeNodes.get(index));
	}

	public long getCurrentLine() {
		return this.currentLine;
	}

	public ArrayList<Long> getBreakpoints() {
		return this.breakpoints;
	}

	public void clearBreakpoints() {
		this.breakpoints.clear();
	}

	public boolean isBreakpoint(long line) {
		return this.breakpoints.contains(line);
	}

	public void addBreakpoint(long line) {
		if (!this.breakpoints.contains(line)) {
			this.breakpoints.add(line);
		}
	}

	public void removeBreakpoint(long line) {
		if (this.breakpoints.contains(line)) {
			this.breakpoints.remove(line);
		}
	}

	private static void sleep() {
		if (Play.getPlay().getActivePerformance().isStepRequest()) {
			return;
		}
		try {
			double speed = Play.getPlay().getActivePerformance()
					.getSimulationSpeed();
			if (speed <= 0) {
				speed = 0.5;
			}
			int value = (int) ((-Math.log(speed) + Math
					.log(Performance.MAX_SPEED)) * 200);
			Thread.sleep(50 + value);
		} catch (InterruptedException exc) {
			Thread.currentThread().interrupt();
		}
	}

	public static void sleepIntern() {
		try {
			double speed = Play.getPlay().getActivePerformance()
					.getSimulationSpeed();
			if (speed <= 0) {
				speed = 0.5;
			}
			int value = (int) ((-Math.log(speed) + Math
					.log(Performance.MAX_SPEED)) * 200);
			Thread.sleep(50 + value);
		} catch (InterruptedException exc) {
			Thread.currentThread().interrupt();
		}
	}

	private boolean checkBreakpoints() {
		if (this.breakpoints.contains(this.currentLine)
				&& this.lastLine != this.currentLine) {
			if (!Play.getPlay().getActivePerformance().isStepRequest()) {
				Play.getPlay().getActivePerformance().setLine(this.currentLine);
			}
			try {
				SwingUtilities.invokeAndWait(new Runnable() {
					public void run() {
						SolistEditor.getSolistEditor().getEditorPanel()
								.markLine(SolistDebugger.this.currentLine - 1);
					}
				});
			} catch (Throwable th) {

			}
			Play.getPlay().getActivePerformance().suspendSimulation();
			return true;
		}
		return false;
	}

	public ArrayList<FrameElement> buildStackFrame(StackTraceElement[] elems) {
		ArrayList<FrameElement> frames = new ArrayList<FrameElement>();
		for (int i = 0; i < elems.length; i++) {
			StackTraceElement elem = elems[i];
			String fileName = elem.getFileName();
			if (fileName.equals(SolistEditor.CLASS_NAME + ".java")) {
				FrameElement f = new FrameElement(elem.getClassName(), elem
						.getMethodName(), elem.getLineNumber());
				frames.add(0, f);
			}
		}
		return frames;
	}

	public ArrayList<VariableInfo> buildVariables(Object... vars) {
		ArrayList<VariableInfo> variables = new ArrayList<VariableInfo>();
		for (int i = 0; i < vars.length; i += 3) {
			variables.add(new VariableInfo((String) vars[i],
					(String) vars[i + 1], vars[i + 2]));
		}
		return variables;
	}

	public void printStack(ArrayList<FrameElement> elems) {
		System.out.println("***");
		for (FrameElement e : elems) {
			System.out.println(e);
		}
		System.out.println("***");
	}

	public void printVars(Object... vars) {
		System.out.println("+++");
		for (int i = 0; i < vars.length; i += 3) {
			System.out.println(vars[i] + " : " + vars[i + 1] + " : "
					+ vars[i + 2]);
		}
		System.out.println("+++");
	}
}
