package de.schmaeck.simulator.model;

import java.io.BufferedReader;
import java.io.FileReader;
import java.io.FileWriter;
import java.util.ArrayList;

import de.schmaeck.struktogrammeditor.HaSE;
import de.schmaeck.struktogrammeditor.model.environment.Environment;

public class Field {

  // pro Zeile (row) wird eine Liste mit Spalten (col) gespeichert
  private ArrayList<ArrayList<Integer>> field;

  // Anzahl an Reihen und Spalten
  private int rows;
  private int cols;

  // Der Hamster
  public Hamster hamster;
  
  private HaSE editor;
  
  // Gespeicherte / Geladene Dateien haben einen Namen.
  private String fileName = null;
  
  private Field backupField = null;
  
  // Kachelzustnde:
  // -1 fr Wand, sonst: Anzahl der Krner
  public static final int WALL = -1;
  public static final int EMPTY = 0;
  
  // Grenzen fr die Felddimensioen
  public static final int MIN = 1;
  public static final int MAX = 100;

  public Field(HaSE editor) {
    this.fileName = null;
    this.backupField = null;
    this.editor = editor;
    this.field = new ArrayList<ArrayList<Integer>>();
    this.rows = 0;
    this.cols = 0;
    this.hamster = new Hamster(this, 3, 0, Hamster.EAST);
    this.setDimension(3, 9);
    this.backupField = new Field(this);
  }
  
  // Kopiert mit ausnahme von backupField
  public Field(Field field) {
    this.backupField = null;
    copyDataFromField(field);
  }
  
  // Kopiert mit ausnahme von backupField
  public void copyDataFromField(Field field) {
    this.fileName = field.fileName;
//    this.backupField = null;
    this.editor = field.editor;
    this.field = new ArrayList<ArrayList<Integer>>(field.rows);
    for (int r = 0; r < field.rows; r++) {
      this.field.add(new ArrayList<Integer>(field.cols));
      for (int c = 0; c < field.cols; c++) {
        this.field.get(r).add(field.getState(r, c));
      }
    }
    this.rows = field.rows;
    this.cols = field.cols;
    this.hamster = new Hamster(this, 
        field.hamster.getRow(), 
        field.hamster.getCol(), 
        field.hamster.getDirection());
    this.hamster.setCornCounter(field.hamster.getCornCounter());
  }
  
  public void setDimension(int rows, int cols) {
    if (cols < 0 || rows < 0) return;

    // 1. berschssige Zeilen lschen oder neue Zeilen anfgen
    if (rows < this.rows) {
      // Zeilen lschen
      for (int i = 0; i < this.rows - rows; i++) {
        this.field.remove(this.field.size() - 1);
      }
      
    } else if (rows > this.rows) {
      // Zeilen hinzufgen
      for (int i = 0; i < rows - this.rows; i++) {
        ArrayList<Integer> newAL = new ArrayList<Integer>(cols);
        for (int j = 0; j < cols; j++) {
          newAL.add(EMPTY);
        }
        this.field.add(newAL);
      }
    }
    
    // 2. Vorhandene Zeilen erweitern oder krzen
    for (int i = 0; i < Math.min(rows, this.rows); i++) {
      if (cols < this.cols) {
        // Kacheln lschen
        for (int j = 0; j < this.cols - cols; j++) {
          this.field.get(i).remove(this.field.get(i).size()-1);
        }
      } else if (cols > this.cols) {
        // Kacheln hinzufgen
        for (int j = 0; j < cols - this.cols; j++) {
          this.field.get(i).add(EMPTY);
        }
      }
    }
    
    this.rows = rows;
    this.cols = cols;
    relocateHamster();
  }
  
  private void relocateHamster() {
    if (hamster.getRow() >= this.rows || hamster.getCol() >= this.cols) {
      for (int row = 0; row < this.rows; row++) {
        for (int col = 0; col < this.cols; col++) {
          if (getState(row, col) != WALL) {
            hamster.locate(row,col);
            return;
          }
        }
      }
    }
    
    // keine freie Kachel gefunden? -> Eine Mauer lschen und Hamster positionieren, 
    // um eine valide Welt zu behalten
    setState(0, 0, EMPTY);
    hamster.locate(0, 0);
  }
  
  
  public int getState(int row, int col) {
    // Auerhalb des Spielfeldes ist alles unpassierbar, also Wand
    if (row < 0 || row >= this.rows || col < 0 || col >= this.cols) return WALL;
    return this.field.get(row).get(col);
  }

  public void setState(int row, int col, int state) {
    if (row < 0 || row >= this.rows || col < 0 || col >= this.cols) return;
    this.field.get(row).set(col, state);
  }
  
  public String toString() {
    String out = "";
    for (int row = 0; row < this.rows; row++) {
      for (int col = 0; col < this.cols; col++) {
        if (hamster.getRow() == row && hamster.getCol() == col) {
          out += hamster.getDirectionChar();
        } else if (getState(row, col) == WALL) {
          out += "#";
        } else if (getState(row, col) == EMPTY) {
          out += ".";
        } else if (getState(row, col) < 10) {
          out += getState(row, col);
        } else {
          out += "+";
        }
      }
      out += "\n";
    }
    out += hamster.toString();
    
    return out;
  }


  public void addCorn (int row, int col) {
    if (this.getState(row, col) != WALL)
        this.setState(row, col, this.getState(row, col) + 1);
  }
  public void subCorn (int row, int col) {
    if (this.getState(row, col) > EMPTY)
      this.setState(row, col, this.getState(row, col) - 1);
  }
  public void setCornCounter(int row, int col, int counter) {
    if (this.getState(row, col) != WALL)
      this.setState(row, col, Math.max(0, counter));
  }
  
  public void toggleWall(int row, int col) {
    
    if (getState(row, col) == WALL) {
      setState(row, col, EMPTY);
    } else if (hamster.getRow() != row || hamster.getCol() != col){
      // Das Hamsterfeld darf keine Mauerbekommen
      setState(row, col, WALL);
    }
  }
  
  // Falls erlaubt: Hamster versetzten. Falls der Hamster schon hier ist:
  // Hamster drehen.
  public void locateHamster(int row, int col, boolean turnLeft) {
    if (hamster.getRow() != row || hamster.getCol() != col) {
      if (getState(row, col) != WALL) {
        hamster.locate(row, col);
      }
    } else if (turnLeft) {
      hamster.linksUm();
    } else {
      hamster.rechtsUm();
    }
  }
  
  // die Environment-Methode filtert Kommentarzeilen heraus und ist hier unbrauchbar 
  private ArrayList<String> loadFieltoAL(String fileName) {
    ArrayList al = new ArrayList<String>();
    String line;
    try {
      FileReader fr = new FileReader(fileName);
      BufferedReader br = new BufferedReader(fr);

      line = br.readLine();
      while (line != null) {
        al.add(line);
        line = br.readLine();
      }

      fr.close();
    } catch (Exception e) {
      return null;
    }
    return al;
  }
  
  public void loadFromFile(String fileName) {
    ArrayList<String> al = loadFieltoAL(fileName);
    if (al == null) return;
    try {
      this.setDimension(Integer.parseInt(al.get(1)), Integer.parseInt(al.get(0)));
      int lineCounter = 2 + this.rows; // erste Zeile mit Krneranzahl 
      for (int row = 0; row < this.rows; row++) {
        for (int col = 0; col < this.cols; col++) {
          switch (al.get(row+2).charAt(col)) {
            case '#': {
              setState(row, col, WALL);
              break;
            }
            case '*': {
              setState(row, col, Integer.parseInt(al.get(lineCounter++)));
              break;
            }
            case '^': {
              setState(row, col, Integer.parseInt(al.get(lineCounter++)));
              this.hamster = new Hamster(this, row, col, Hamster.NORTH);
              break;
            }
            case '>': {
              setState(row, col, Integer.parseInt(al.get(lineCounter++)));
              this.hamster = new Hamster(this, row, col, Hamster.EAST);
              break;
            }
            case 'v': {
              setState(row, col, Integer.parseInt(al.get(lineCounter++)));
              this.hamster = new Hamster(this, row, col, Hamster.SOUTH);
              break;
            }
            case '<': {
              setState(row, col, Integer.parseInt(al.get(lineCounter++)));
              this.hamster = new Hamster(this, row, col, Hamster.WEST);
              break;
            }
            default: { //case ' ': {
              setState(row, col, EMPTY);
              break;
            }
          }
        }
      }
      this.hamster.setCornCounter(Integer.parseInt(al.get(lineCounter)));
      this.fileName = fileName;
      this.backupField = new Field(this);

    } catch (Exception ex) {
      
    }
  }
  
  public ArrayList<String> exportToAL() {
    ArrayList<String> out = new ArrayList<String>();
    out.add("" + this.cols);
    out.add("" + this.rows);
    for (int row = 0; row < this.rows; row++) {
      String line = "";
      for (int col = 0; col < this.cols; col++) {
        if (getState(row, col) == WALL) {
            line += '#';
        } else if (hamster.getRow() == row && hamster.getCol() == col) {
          line += hamster.getDirectionChar();
          out.add("" + getState(row, col));
        } else if (getState(row, col) > 0) {
          line += '*';
          out.add("" + getState(row, col));
        } else {
          line += ' ';
        }
      }
      out.add(row+2, line);
    }
    out.add("" + hamster.getCornCounter());
    
    return out;
  }
  
  public void saveToFile(String fileName) {
    try {
      FileWriter fw = new FileWriter(fileName);
      ArrayList<String> al = this.exportToAL();
      for (int i = 0; i < al.size(); i++) {
        fw.write(al.get(i) + this.editor.environment.getNl());
      }
      fw.close();
      this.fileName = fileName;
      this.backupField = new Field(this);
    } catch (Exception ex) {
      ex.printStackTrace();
    }
  }

  public int getCols () {
    return this.cols;
  }
  public int getRows () {
    return this.rows;
  }

  public void setFileName (String fileName) {
    this.fileName = fileName;
  }

  public String getFileName () {
    return this.fileName;
  }

  public void restore () {
    if (this.backupField != null) {
      this.copyDataFromField(this.backupField);
    }
    
  }
}
