package theater_intern;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;

import simulation.SimulationManager;
import theater.Actor;
import theater.Component;
import theater.PixelArea;
import theater.Stage;
import theater.TheaterImage;
import util.Utils;

/**
 * Interne Umsetzung einer Bhne
 * 
 * @author Dietrich Boles, Uni Oldenburg
 * @version 1.0 (12.11.2008)
 * 
 */
public class IStage implements Serializable {

	private static HashMap<Stage, IStage> stageMap = new HashMap<Stage, IStage>();
	private static HashMap<IStage, Stage> iStageMap = new HashMap<IStage, Stage>();

	public static void newStageIStage(Stage stage, IStage iStage) {
		IStage.stageMap.put(stage, iStage);
		IStage.iStageMap.put(iStage, stage);
	}

	public static Stage getStage(IStage iStage) {
		return IStage.iStageMap.get(iStage);
	}

	public static IStage getIStage(Stage stage) {
		return IStage.stageMap.get(stage);
	}

	transient private ArrayList<IComponent> allComponents;
	transient private IComponent solist;

	private TheaterImage background;
	private int cellSize;
	private int noOfCols;
	private int noOfRows;
	private boolean tiled;
	private boolean handlingKeyEvents;
	private boolean handlingMouseEvents;

	public IStage(int noOfCols, int noOfRows, int cellSize) {
		this.noOfCols = noOfCols;
		this.noOfRows = noOfRows;
		this.cellSize = cellSize;
		this.tiled = true;
		this.allComponents = new ArrayList<IComponent>();
		this.background = null; // new TheaterImage(noOfCols, noOfRows);
		this.handlingKeyEvents = false;
		this.handlingMouseEvents = false;
		this.solist = null;
	}

	public static void setStage(IStage stage) {
		SimulationManager.getSimulationManager().setStage(stage);
	}

	public void add(IComponent comp, int col, int row) {
		if (this.allComponents.contains(comp) || comp.getStage() != null) {
			return;
		}
		comp.setStage(this);
		Component component = IComponent.getComp(comp);
		component.setLocation(col, row);
		this.allComponents.add(comp);
		
		if (component instanceof Actor) {
			if (this.solist != null) {
				this.removeI(this.solist);
			}
			this.solist = comp;
		}
		
		component.addedToStage(IStage.getStage(this));


		TheaterObservable.getObservable().stateChange();
	}

	public void remove(IComponent comp) {
		if (!this.allComponents.contains(comp)) {
			return;
		}
		if (solist != null && comp == solist) {
			return;
		}
		this.allComponents.remove(comp);
		comp.setStage(null);
		Component component = IComponent.getComp(comp);
		component.removedFromStage(IStage.getStage(this));
		TheaterObservable.getObservable().stateChange();

	}

	private void removeI(IComponent comp) {
		if (!this.allComponents.contains(comp)) {
			return;
		}

		this.allComponents.remove(comp);
		comp.setStage(null);
		Component component = IComponent.getComp(comp);
		component.removedFromStage(IStage.getStage(this));
		TheaterObservable.getObservable().stateChange();
	}

	public void remove(List<Component> comps) {
		for (Component comp : comps) {
			IComponent icomp = IComponent.getIComp(comp);
			if (solist != null && icomp == solist) {
				continue;
			}
			if (this.allComponents.contains(icomp)) {
				this.allComponents.remove(icomp);
				icomp.setStage(null);
				Component component = IComponent.getComp(icomp);
				component.removedFromStage(IStage.getStage(this));
			}
		}
		TheaterObservable.getObservable().stateChange();
	}

	public int getNumberOfColumns() {
		return this.noOfCols;
	}

	public int getNumberOfRows() {
		return this.noOfRows;
	}

	public int getCellSize() {
		return this.cellSize;
	}

	public void setSize(int noOfCols, int noOfRows) {
		this.noOfCols = noOfCols;
		this.noOfRows = noOfRows;
		TheaterObservable.getObservable().stateChange();
	}

	public void setSize(int noOfCols, int noOfRows, int cellSize) {
		this.noOfCols = noOfCols;
		this.noOfRows = noOfRows;
		this.cellSize = cellSize;
		TheaterObservable.getObservable().stateChange();
	}

	public boolean isTiled() {
		return this.tiled;
	}

	public final void setBackground(TheaterImage image) {
		if (this.background != image) {
			this.background = image;
			TheaterObservable.getObservable().stateChange();
		}
	}

	public final void setBackground(String filename)
			throws IllegalArgumentException {
		this.setBackground(new TheaterImage(filename));
	}

	public TheaterImage getBackground() {
		return this.background;
	}

	public void paint() {
		TheaterObservable.getObservable().importantStateChange();
	}

	public List<IComponent> getComponentsInPaintOrder() {
		Collections.sort(this.allComponents, new IComponentComparator());
		return new ArrayList<IComponent>(this.allComponents);
	}

	public int numberOfComponents() {
		return this.allComponents.size();
	}

	public List<IComponent> getAllIComponents() {
		return new ArrayList<IComponent>(this.allComponents);
	}

	static java.awt.Point toAWTPoint(theater.Point p) {
		return new java.awt.Point(p.getX(), p.getY());
	}

	public List<Component> getComponents(Class<?>... classes) {
		ArrayList<Component> comps = new ArrayList<Component>(this.allComponents.size());
		for (IComponent iComp : this.allComponents) {
			Component comp = IComponent.getComp(iComp);
			if (IStage.isInstanceOf(comp, classes)) {
				comps.add(comp);
			}
		}
		return comps;
	}

	public List<Component> getComponentsAt(int col, int row,
			Class<?>... classes) {
		ArrayList<Component> comps = new ArrayList<Component>(this.allComponents.size());
		for (IComponent iComp : this.allComponents) {
			if (iComp.getRow() == row && iComp.getColumn() == col) {
				Component comp = IComponent.getComp(iComp);
				if (IStage.isInstanceOf(comp, classes)) {
					comps.add(comp);
				}
			}
		}
		return comps;
	}

	public List<Component> getComponentsInside(PixelArea area,
			Class<?>... classes) {
		ArrayList<Component> comps = new ArrayList<Component>(this.allComponents.size());
		for (IComponent iComp : this.allComponents) {
			Component comp = IComponent.getComp(iComp);
			if (IStage.isInstanceOf(comp, classes) && iComp.isInside(area)) {
				comps.add(comp);
			}
		}
		return comps;
	}

	public List<Component> getIntersectingComponents(PixelArea area,
			Class<?>... classes) {
		ArrayList<Component> comps = new ArrayList<Component>(this.allComponents.size());
		for (IComponent iComp : this.allComponents) {
			Component comp = IComponent.getComp(iComp);
			if (IStage.isInstanceOf(comp, classes) && iComp.intersects(area)) {
				comps.add(comp);
			}
		}
		return comps;
	}

	static boolean isInstanceOf(Component comp, Class<?>... classes) {
		if (classes == null || classes.length == 0) {
			return true;
		}
		for (Class<?> c : classes) {
			if (Utils.dynInstanceof(comp, c)) {
				return true;
			}
		}
		return false;
	}

	public boolean isHandlingKeyEvents() {
		return this.handlingKeyEvents;
	}

	public void setHandlingKeyEvents(boolean handlingKeyEvents) {
		this.handlingKeyEvents = handlingKeyEvents;
	}

	public boolean isHandlingMouseEvents() {
		return this.handlingMouseEvents;
	}

	public void setHandlingMouseEvents(boolean handlingMouseEvents) {
		this.handlingMouseEvents = handlingMouseEvents;
	}

	public IComponent getSolist() {
		return this.solist;
	}

	// called after deserialization
	public void _internal_reinit() {
		this.allComponents = new ArrayList<IComponent>();
		this.solist = null;
	}

	public void _internal_add(Component comp) {
		IComponent iComp = IComponent.getIComp(comp);
		this.allComponents.add(iComp);
		iComp.setStage(this);

		if (comp instanceof Actor) {
			this.solist = iComp;
		}
	}

}
