package editor;

import java.awt.Color;
import java.awt.EventQueue;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.event.MouseEvent;
import java.util.ArrayList;
import java.util.Iterator;

import javax.swing.JTextPane;
import javax.swing.SwingUtilities;
import javax.swing.text.DefaultStyledDocument;
import javax.swing.text.Element;
import javax.swing.text.StyleConstants;

import model.Play;
import util.PropertyManager;
import util.ResourceManager;
import view.BreakpointsPopupMenu;

import debugger.SolistDebugger;

/**
 * Ein TextPane, in dem die Zeilennummern des Editors angezeigt werden.
 * 
 * @author Dietrich Boles, Uni Oldenburg
 * @version 1.0 (12.11.2008)
 * 
 */
public class LineNumberPanel extends JTextPane {

	LineNumberDocument doc;
	Editor editor;
	ArrayList<Long> lines;
	int noOfLines;

	public LineNumberPanel(Editor editor) {
		this.editor = editor;
		this.setEditable(false);
		this.doc = new LineNumberDocument(editor, this);
		lines = new ArrayList<Long>();
		this.setDocument(this.doc);
		this.setBackground(new Color(230, 230, 230));
		this.noOfLines = 0;
	}

	public void setNumberOfLines(int noOfLines) {
		this.noOfLines = noOfLines;

		StringBuffer text = new StringBuffer();
		for (int i = 1; i <= noOfLines; i++) {
			text.append(i + "\n");
		}
		ArrayList<Long> breakpoints = SolistDebugger.getSolistDebugger()
				.getBreakpoints();
		Iterator<Long> iter = breakpoints.iterator();
		while (iter.hasNext()) {
			long b = iter.next();
			if (b > noOfLines) {
				iter.remove();
			}
		}

		this.setText(text.toString());
		markLines();
	}

	protected void processMouseEvent(MouseEvent e) {
		super.processMouseEvent(e);
		if (editor != SolistEditor.getSolistEditor().getEditor()) {
			return;
		}
		if (e.getID() == MouseEvent.MOUSE_PRESSED && e.getClickCount() == 2) {

			if (PropertyManager.getPropertyManager().isDebuggingOn()) {
				long line = getLineNumber(e.getY());
				if (line <= noOfLines) {
					if (SolistDebugger.getSolistDebugger().isBreakpoint(line)) {
						SolistDebugger.getSolistDebugger().removeBreakpoint(
								line);
						Play
								.getPlay()
								.getPlayFrame()
								.getMessagePanel()
								.writeInfo(
										ResourceManager.getResourceManager()
												.getValue("breakpoint.removed"));
					} else {
						SolistDebugger.getSolistDebugger().addBreakpoint(line);
						Play.getPlay().getPlayFrame().getMessagePanel()
								.writeInfo(
										ResourceManager.getResourceManager()
												.getValue("breakpoint.added"));
					}
					markLines();
				}
			}

		} else if (e.isPopupTrigger()) {
			if (PropertyManager.getPropertyManager().isDebuggingOn()) {
				long line = getLineNumber(e.getY());
				BreakpointsPopupMenu menu = new BreakpointsPopupMenu(line);
				menu.show(e.getComponent(), e.getX(), e.getY());
			}
		}
	}

	public void markLines() {
		ArrayList<Long> breakpoints = SolistDebugger.getSolistDebugger()
				.getBreakpoints();
		lines.clear();
		for (long breakpoint : breakpoints) {
			markLine(breakpoint - 1);
		}
		if (EventQueue.isDispatchThread()) {
			this.repaint();
		} else {
			try {
				SwingUtilities.invokeAndWait(new Runnable() {
					public void run() {
						repaint();
					}
				});
			} catch (Throwable e) {
				e.printStackTrace();
			}
		}
	}

	public void markLine(long line) {
		long iLine = -1;
		Element root = this.getDocument().getDefaultRootElement();
		if (root != null) {
			Element elem = root.getElement((int) line);
			if (elem != null) {
				this.setCaretPosition(elem.getStartOffset());
				iLine = line;
			} else {
				elem = root.getElement(root.getElementCount() - 1);
				if (elem != null) {
					this.setCaretPosition(elem.getStartOffset());
					iLine = root.getElementCount() - 1;
				}
			}
		}
		if (iLine != -1 && !lines.contains(iLine)) {
			lines.add(iLine);
		}
	}

	public void paintComponent(Graphics g) {
		if (lines.size() > 0) {
			g.setColor(this.getBackground());
			g.fillRect(0, 0, this.getWidth(), this.getHeight());
			Element root = this.getDocument().getDefaultRootElement();
			for (long line : lines) {
				int y = 0;
				try {
					y = (int) this.modelToView(
							root.getElement((int) line).getStartOffset())
							.getY();
					g.setColor(new Color(191, 160, 247));
					g.fillRect(0, y, this.getWidth(), doc.lineHeight);
				} catch (Throwable e) {
				}
			}
			this.setOpaque(false);
			super.paintComponent(g);
			this.setOpaque(true);
		} else {
			super.paintComponent(g);
		}
	}

	public int getLineNumber(int y) {
		if (y < 3) {
			return 1;
		}
		y = y - 3;
		int posLine;
		int lineHeight = this.getFontMetrics(this.getFont()).getHeight();
		posLine = (y / lineHeight) + 1;
		return posLine;
	}

}

class LineNumberDocument extends DefaultStyledDocument {
	Editor editor;
	LineNumberPanel lPanel;
	int fontSize;
	int lineHeight;

	public LineNumberDocument(Editor editor, LineNumberPanel p) {
		this.editor = editor;
		this.lPanel = p;
		this.fontSize = editor.getFontSize();
		initStyles(this.fontSize);
	}

	protected void initStyles(int size) {
		this.addStyle("plain", this.getStyle("default"));
		StyleConstants.setFontFamily(this.getStyle("plain"), "Monospaced");
		StyleConstants.setFontSize(this.getStyle("plain"), size);
		this.setParagraphAttributes(0, this.getLength(),
				this.getStyle("plain"), true);
		this.setCharacterAttributes(0, this.getLength(),
				this.getStyle("plain"), true);
		Font f = new Font("Monospaced", Font.PLAIN, size);
		lPanel.setFont(f);
		FontMetrics fm = lPanel.getFontMetrics(f);
		lineHeight = fm.getHeight();
	}

	public void changeFontSize(int size) {
		this.removeStyle("plain");
		this.fontSize = size;
		initStyles(size);
	}

}
