package editor;

import java.awt.Color;
import java.awt.Font;
import java.awt.FontMetrics;
import java.util.Iterator;
import java.util.LinkedList;

import javax.swing.SwingUtilities;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;
import javax.swing.text.BadLocationException;
import javax.swing.text.DefaultStyledDocument;
import javax.swing.text.StyleConstants;
import javax.swing.text.TabSet;

/**
 * Speziell fr Java-Editoren entwickeltes Document.
 * 
 * @author Dietrich Boles, Uni Oldenburg
 * @version 1.0 (12.11.2008)
 * 
 */
public class JavaDocument extends DefaultStyledDocument implements
		DocumentListener {
	protected Editor editor;
	boolean highlighting;
	int lineHeight;

	protected LinkedList<JavaToken> tokens;

	protected JavaLexer scanner;

	public JavaDocument(Editor editor, EditorPanel panel) {
		this.editor = editor;
		this.scanner = new JavaLexer();
		this.addDocumentListener(this);
		initStyles(panel);
	}

	protected void initStyles(EditorPanel panel) {
		this.addStyle("plain", this.getStyle("default"));

		StyleConstants.setFontFamily(this.getStyle("plain"), "Monospaced");
		StyleConstants
				.setFontSize(this.getStyle("plain"), editor.getFontSize());
		
		TabSet set = panel.calcTabs(4, this.getFont(this.getStyle("plain")));
		StyleConstants.setTabSet(this.getStyle("plain"), set);
		
		this.setParagraphAttributes(0, this.getLength(),
				this.getStyle("plain"), true);

		this.addStyle("keyword", this.getStyle("plain"));
		StyleConstants.setBold(this.getStyle("keyword"), true);
		StyleConstants.setForeground(this.getStyle("keyword"), Color.MAGENTA
				.darker().darker());

		this.addStyle("comment", this.getStyle("plain"));
		StyleConstants.setForeground(this.getStyle("comment"), new Color(63,
				127, 95));

		this.addStyle("literal", this.getStyle("plain"));
		StyleConstants.setForeground(this.getStyle("literal"), Color.BLUE);
		this.tokens = new LinkedList<JavaToken>();
		
		Font f = this.getFont(this.getStyle("plain"));
		FontMetrics fm = panel.getFontMetrics(f);
		lineHeight = fm.getHeight();
	}

	public void changeFontSize(int size) {
		this.removeStyle("plain");
		this.removeStyle("keyword");
		this.removeStyle("comment");
		this.removeStyle("literal");
		
		initStyles(editor.getEditorPanel());
		
		SwingUtilities
				.invokeLater(new RunRehighlight(0, this.getLength(), this));
	}

	public void removeDocListener() {
		this.removeDocumentListener(this);
	}

	public void addDocListener() {
		this.addDocumentListener(this);
		SwingUtilities
				.invokeLater(new RunRehighlight(0, this.getLength(), this));
	}

	JavaToken first() {
		return this.tokens.getFirst();
	}

	int copyAllBefore(int pos, LinkedList<JavaToken> newTokens) {
		int start = 0;
		while (!this.tokens.isEmpty()
				&& this.first().getStart() + this.first().getText().length() < pos) {
			start = this.first().getStart() + this.first().getText().length();
			newTokens.addLast(this.tokens.removeFirst());
		}
		return start;
	}

	public JavaToken getTokenAt(int pos) {
		for (Iterator<JavaToken> iter = this.tokens.iterator(); iter.hasNext();) {
			JavaToken token = iter.next();
			if (token.getStart() >= pos) {
				return token;
			}
		}
		return null;
	}

	boolean exists(JavaToken t, int offset) {
		while (!this.tokens.isEmpty()) {
			if (this.first().getStart() + offset == t.getStart()) {
				return true;
			} else if (this.first().getStart() + offset < t.getStart()) {
				this.tokens.removeFirst();
			} else {
				return false;
			}
		}
		return false;
	}

	void copyAllAfter(LinkedList<JavaToken> newTokens, int offset) {
		while (!this.tokens.isEmpty()) {
			JavaToken t = this.first();
			t.setStart(t.getStart() + offset);
			newTokens.addLast(this.tokens.removeFirst());
		}
	}

	void setAttributes(JavaToken t) {
		if (t.getType() == JavaLexer.COMMENT) {
			this.setCharacterAttributes(t.getStart(), t.getText().length(),
					this.getStyle("comment"), true);
		} else if (t.getType() == JavaLexer.KEYWORD) {
			this.setCharacterAttributes(t.getStart(), t.getText().length(),
					this.getStyle("keyword"), true);
		} else if (t.getType() == JavaLexer.LITERAL) {
			this.setCharacterAttributes(t.getStart(), t.getText().length(),
					this.getStyle("literal"), true);
		} else {
			this.setCharacterAttributes(t.getStart(), t.getText().length(),
					this.getStyle("plain"), true);
		}
	}

	public boolean isHighlighting() {
		return this.highlighting;
	}

	public void rehighlight(int pos, int len) {
		this.highlighting = true;
		LinkedList<JavaToken> newTokens = new LinkedList<JavaToken>();
		int start = this.copyAllBefore(pos, newTokens);

		try {
			this.scanner.init(0, start, this.getText(start, this.getLength()
					- start));
		} catch (BadLocationException e1) {
			// TODO should not happen
		}
		while (this.scanner.ready()) {
			JavaToken t = this.scanner.nextToken();
			if (t == null) {
				break;
			}
			if (t.getStart() > pos && this.exists(t, len)) {
				this.copyAllAfter(newTokens, len);
				break;
			}
			this.setAttributes(t);
			newTokens.addLast(t);
		}
		this.tokens = newTokens;

		this.highlighting = false;
	}
	
	public void rehighlight() {
		rehighlight(0, this.getLength());
	}

	public void changedUpdate(DocumentEvent e) {
	}

	public void insertUpdate(DocumentEvent e) {
		SwingUtilities.invokeLater(new RunRehighlight(e.getOffset(), e
				.getLength(), this));
	}

	public void removeUpdate(DocumentEvent e) {
		SwingUtilities.invokeLater(new RunRehighlight(e.getOffset(), -e
				.getLength(), this));
	}
}