package theater_intern;

import java.applet.Applet;
import java.applet.AudioClip;
import java.awt.EventQueue;
import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.concurrent.locks.ReentrantLock;

import model.Play;
import simulation.SimulationManager;
import sun.audio.AudioPlayer;
import sun.audio.AudioStream;
import theater.Performance;
import debugger.SolistDebugger;

/**
 * Interne Umsetzung einer Performance
 * 
 * @author Dietrich Boles, Uni Oldenburg
 * @version 1.0 (12.11.2008)
 * 
 */
public class IPerformance {

	/***************************************************************************
	 * * * * * * * * * * * * * * Umformungsmethoden
	 */
	private static HashMap<Performance, IPerformance> perfMap = new HashMap<Performance, IPerformance>();
	private static HashMap<IPerformance, Performance> iPerfMap = new HashMap<IPerformance, Performance>();

	public static void newPerfIPerf(Performance perf, IPerformance iPerf) {
		IPerformance.perfMap.put(perf, iPerf);
		IPerformance.iPerfMap.put(iPerf, perf);
	}

	public static Performance getPerformance(IPerformance iPerf) {
		return IPerformance.iPerfMap.get(iPerf);
	}

	public static IPerformance getIPerformance(Performance perf) {
		return IPerformance.perfMap.get(perf);
	}

	private static IPerformance defaultPerformance = null;

	/***************************************************************************
	 * * * * * * * * * * * * * * Performancemethoden
	 */

	final public static int STOPPED = 0;
	final public static int RUNNING = 2;
	final public static int SUSPENDED = 3;

	private AudioClip clip;
	private int state;
	private int speed;
	private int freeze;
	private long line;
	private int number;

	private boolean suspendRequest;
	private boolean stopRequest;
	private boolean stepRequest;
	private Object syncObject = new Object();

	FileInputStream inputStream;
	AudioStream audioStream;
	List<InputStream> audios;

	public static IPerformance getDefaultPerformance() {
		if (IPerformance.defaultPerformance == null) {
			Performance defPerformance = new Performance();
			IPerformance.defaultPerformance = IPerformance
					.getIPerformance(defPerformance);
		}
		return IPerformance.defaultPerformance;
	}

	public IPerformance() {
		this.clip = null;
		this.state = IPerformance.STOPPED;
		this.speed = Performance.DEF_SPEED;
		this.freeze = 0;
		this.suspendRequest = false;
		this.stopRequest = false;
		this.stepRequest = false;
		this.line = -1;
		this.number = -1;
		this.audios = new ArrayList<InputStream>();
		this.inputStream = null;
		this.audioStream = null;
		// this.syncObject = new Object();
	}

	public void startSimulation() {
		if (this.state != IPerformance.STOPPED) {
			return;
		}
		this.newLock();
		this.suspendRequest = false;
		this.stopRequest = false;
		this.stepRequest = false;
		this.line = -1;
		this.number = -1;
		// this.syncObject = new Object();
		this.state = IPerformance.RUNNING;
		forceUnfreeze();
		IPerformance.getPerformance(this).started();
		TheaterObservable.getObservable().performanceStateChange();
	}

	public void stopSimulation() {
		if (this.state == IPerformance.STOPPED) {
			return;
		}
		this.state = IPerformance.STOPPED;
		IPerformance.getPerformance(this).stopped();
		forceUnfreeze();
		stopSounds();
		TheaterObservable.getObservable().performanceStateChange();

		if (!this.stopRequest) {
			SimulationManager.getSimulationManager().handleAPIStop();
		}
		this.stepRequest = false;

	}

	public void suspendSimulation() {
		if (this.state != IPerformance.RUNNING) {
			return;
		}
		this.state = IPerformance.SUSPENDED;
		IPerformance.getPerformance(this).suspended();
		TheaterObservable.getObservable().performanceStateChange();

		if (!this.suspendRequest && !this.stepRequest) {
			SimulationManager.getSimulationManager().handleAPISuspend();
		}

	}

	public void resumeSimulation() {
		if (this.state != IPerformance.SUSPENDED) {
			return;
		}
		this.state = IPerformance.RUNNING;
		IPerformance.getPerformance(this).resumed();
		TheaterObservable.getObservable().performanceStateChange();

	}

	public void resetSimulation() {
		if (!this.simulationStopped()) {
			try {
				this.stopSimulation();
			} catch (Throwable t) {

			}
		}
		Play.getPlay().reset();
	}

	public boolean simulationRunning() {
		return this.state == IPerformance.RUNNING;
	}

	public boolean simulationStopped() {
		return this.state == IPerformance.STOPPED;
	}

	public boolean simulationSuspended() {
		return this.state == IPerformance.SUSPENDED;
	}

	public void checkStop() {

		if (this.simulationStopped()
				|| Thread.currentThread() != SimulationManager
						.getSimulationManager().getSimulator()) {
			return;
		}

		if (this.lock.isHeldByCurrentThread()) {
			return;
		}

		synchronized (this.syncObject) {

			if (this.stopRequest) {
				this.suspendRequest = false;
				this.stopRequest = false;
				throw new StopException();
			}
			while (this.suspendRequest) {
				try {
					this.suspendSimulation();
					SimulationManager.getSimulationManager().setPause();
					this.syncObject.wait();
					if (this.stopRequest) {
						this.stepRequest = false;
						this.suspendRequest = false;
						this.stopRequest = false;
						SimulationManager.getSimulationManager().setStop();
						throw new StopException();
					}
					this.resumeSimulation();
					if (this.stepRequest) {
						break;
					}
				} catch (InterruptedException e) {
				}
			}
		}

	}

	public void setLine(long line) {
		this.line = line;
	}

	public void checkStep(int curNumber) {

		if (this.simulationStopped()
				|| Thread.currentThread() != SimulationManager
						.getSimulationManager().getSimulator()) {
			return;
		}

		if (this.lock.isHeldByCurrentThread()) {
			return;
		}

		long cLine = SolistDebugger.getSolistDebugger().getCurrentLine();
		if (cLine == this.line && this.number != curNumber) {
			return;
		}

		this.line = cLine;
		this.number = curNumber;

		synchronized (this.syncObject) {
			while (this.stepRequest) {
				try {
					this.suspendSimulation();
					this.stepRequest = false;
					this.syncObject.wait();
					if (this.stopRequest) {
						this.stepRequest = false;
						this.suspendRequest = false;
						this.stopRequest = false;
						SimulationManager.getSimulationManager().setStop();
						throw new StopException();
					}
					this.resumeSimulation();
					if (this.stepRequest) {
						break;
					}
				} catch (InterruptedException e) {
				}
			}
		}

	}

	public void suspendRequest() {
		synchronized (this.syncObject) {
			this.suspendRequest = true;
		}

	}

	public void stopRequest() {
		synchronized (this.syncObject) {
			this.stopRequest = true;
			this.stepRequest = false;
			this.syncObject.notifyAll();
		}
	}

	public void resumeRequest() {
		synchronized (this.syncObject) {
			this.suspendRequest = false;
			this.stopRequest = false;
			this.syncObject.notifyAll();
		}
	}

	public void stepRequest() {
		synchronized (this.syncObject) {
			this.stepRequest = true;
			this.resumeRequest();
		}
	}

	public void freeze() {
		this.freeze++;
	}

	public void unfreeze() {
		if (this.freeze > 0) {
			this.freeze--;
		}
		TheaterObservable.getObservable().importantStateChange();
	}

	public void forceUnfreeze() {
		if (this.freeze != 0) {
			this.freeze = 0;
			TheaterObservable.getObservable().importantStateChange();
		}
	}

	public int getFreeze() {
		return this.freeze;
	}

	public boolean frozen() {
		return this.freeze > 0;
	}

	public void playSound2(String soundFile) {
		String dir = Play.getPlay().getDirectory();
		File f = new File(dir + File.separatorChar + "sounds"
				+ File.separatorChar + soundFile);
		if (!f.exists()) {
			throw new IllegalArgumentException();
		}

		try {
			URL url = f.toURI().toURL();
			this.clip = Applet.newAudioClip(url);
			this.clip.play(); // dauert manchmal sehr lange
		} catch (MalformedURLException e) {
			throw new IllegalArgumentException();
		}
	}

	public void playSound(String soundFile) {
		String dir = Play.getPlay().getDirectory();
		File f = new File(dir + File.separatorChar + "sounds"
				+ File.separatorChar + soundFile);
		if (!f.exists()) {
			throw new IllegalArgumentException();
		}
		try {
			this.inputStream = new FileInputStream(f.getAbsolutePath());
			this.audioStream = new AudioStream(this.inputStream);
			AudioPlayer.player.start(this.audioStream);
			audios.add(this.audioStream);
			audios.add(this.inputStream);
		} catch (Exception exc) {
		}
	}

	public void stopSounds() {
		new SoundsStop(new ArrayList<InputStream>(this.audios)).start();
		audios = new ArrayList<InputStream>();
	}

	public int getSimulationSpeed() {
		return this.speed;
	}

	public void setSimulationSpeedAPI(int newSpeed) {
		if (newSpeed != this.speed) {
			this.speed = newSpeed;
			IPerformance.getPerformance(this).speedChanged(this.speed);
			if (Play.getPlay().getPlayFrame() != null) {
				Play.getPlay().getPlayFrame().setSpeed(newSpeed);
			}
		}
	}

	public void setSimulationSpeed(int newSpeed) {
		if (newSpeed != this.speed) {
			this.speed = newSpeed;
		}
	}

	// locking

	private ReentrantLock lock = new SolistReentrantLock(true);

	public void newLock() {
		this.lock = new SolistReentrantLock(true);
	}

	public void lock() {
		if (!this.simulationRunning() || EventQueue.isDispatchThread()) {
			// || Thread.currentThread() != SimulationManager
			// .getSimulationManager().getSimulator()) {
			return;
		}
		this.lock.lock();
	}

	public void unlock() {
		// if (!this.simulationRunning() || EventQueue.isDispatchThread()) {
		// return;
		// }
		this.lock.unlock();
	}

	public void forceLock() {
		this.lock.lock();
	}

	public void forceUnlock() {
		this.lock.unlock();
	}

	public boolean isStepRequest() {
		return this.stepRequest;
	}

}

class SolistReentrantLock extends ReentrantLock {

	private int calls;

	SolistReentrantLock(boolean isFair) {
		super(isFair);
		this.calls = 0;
	}

	public void lock() {
		super.lock();
		this.calls++;
	}

	public void unlock() {
		if (this.getHoldCount() > 0) {
			super.unlock();
		}
	}

}

class SoundsStop extends Thread {

	ArrayList<InputStream> audios;

	SoundsStop(ArrayList<InputStream> audios) {
		this.audios = audios;
	}

	public void run() {
		try {
			Thread.sleep(3000);
		} catch (InterruptedException exc) {
		}
		for (int i = 0; i < audios.size(); i += 2) {
			stopSound((AudioStream) audios.get(i), (FileInputStream) audios
					.get(i + 1));
		}
	}

	static void stopSound(AudioStream aStream, FileInputStream iStream) {
		try {
			if (aStream != null) {
				AudioPlayer.player.stop(aStream);
				aStream.close();
			}
		} catch (Throwable th) {
		}
		try {
			if (iStream != null) {
				iStream.close();
			}
		} catch (Throwable th) {
		}
	}
}
