package editor;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.EventQueue;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStreamReader;

import javax.swing.JTextPane;
import javax.swing.SwingUtilities;
import javax.swing.event.UndoableEditEvent;
import javax.swing.event.UndoableEditListener;
import javax.swing.text.AttributeSet;
import javax.swing.text.BadLocationException;
import javax.swing.text.DocumentFilter;
import javax.swing.text.Element;
import javax.swing.text.StyledEditorKit;
import javax.swing.text.TabSet;
import javax.swing.text.TabStop;

import debugger.SolistDebugger;

/**
 * Die TextPane, in der der Benutzer Text eingeben kann.
 * 
 * @author Dietrich Boles, Uni Oldenburg
 * @version 1.0 (12.11.2008)
 * 
 */
public class EditorPanel extends JTextPane implements UndoableEditListener {

	Editor editor;
	JavaDocument doc;
	InputFilter inputFilter;
	int numberOfLines;
	long line;

	public EditorPanel(Editor editor) {
		this.editor = editor;
		this.setEditorKit(new StyledEditorKit());
		if (editor.isJava) {
			this.doc = new JavaDocument(editor, this);
		} else {
			this.doc = new TextDocument(editor, this);
		}
		this.setDocument(this.doc);
		this.setBackground(Color.WHITE);
		this.numberOfLines = 0;
		this.line = -1;
		this.addMouseListener(new MouseAdapter() {
			public void mousePressed(MouseEvent e) {
				EditorPanel.this.removeLineHighlight();
				EditorPanel.this.repaint();
			}
		});
	}

	public JavaDocument getJavaDocument() {
		return this.doc;
	}

	public void setSize(Dimension d) {
		if (d.width < this.getParent().getSize().width) {
			d.width = this.getParent().getSize().width;
		}
		// kein LineWrap
		super.setSize(d);
	}

	public boolean getScrollableTracksViewportWidth() {
		return false; // kein LineWrap
	}

	public void loadFile(String file) {
		FileReader fr = null;
		try {
			File f = new File(file);
			fr = new FileReader(f);
			// fr = new InputStreamReader(
			// new FileInputStream(f), "UTF-8");
			char[] buffer = new char[(int) f.length()];
			fr.read(buffer);
			this.doc.setDocumentFilter(null);
			this.doc.removeDocListener();
			this.setText(new String(buffer));
			this.doc.addDocListener();

			this.inputFilter = new InputFilter(this);
			this.doc.setDocumentFilter(this.inputFilter);
			this.setCaretPosition(0);
			this.calcNumberOfLines();
			this.editor.getUndoManager().discardAllEdits();
			this.editor.setChanged(false);
		} catch (Throwable exc) {
			exc.printStackTrace();
		} finally {
			if (fr != null) {
				try {
					fr.close();
				} catch (IOException e) {
				}
			}
		}
	}

	public void loadFile(File f) {
		FileReader fr = null;
		try {
			fr = new FileReader(f);
			// fr = new InputStreamReader(
			// new FileInputStream(f), "UTF-8");
			char[] buffer = new char[(int) f.length()];
			fr.read(buffer);
			this.doc.setDocumentFilter(null);
			this.doc.removeDocListener();
			this.setText(new String(buffer));
			this.doc.addDocListener();

			this.inputFilter = new InputFilter(this);
			this.doc.setDocumentFilter(this.inputFilter);
			this.setCaretPosition(0);
			this.calcNumberOfLines();
			this.editor.getUndoManager().discardAllEdits();
			SolistDebugger.getSolistDebugger().clearBreakpoints();
			editor.getLineNumberPanel().markLines();
			this.editor.setChanged(false);
		} catch (Throwable exc) {
			exc.printStackTrace();
		} finally {
			if (fr != null) {
				try {
					fr.close();
				} catch (IOException e) {
				}
			}
		}
	}

	public TabSet calcTabs(int charactersPerTab, Font font) {
		FontMetrics fm = this.getFontMetrics(font);
		int charWidth = fm.charWidth('w');
		int tabWidth = charWidth * charactersPerTab;

		TabStop[] tabs = new TabStop[100];
		tabs[0] = new TabStop(tabWidth);

		for (int j = 1; j < tabs.length; j++) {
			tabs[j] = new TabStop(tabs[j - 1].getPosition() + tabWidth);
		}

		TabSet tabSet = new TabSet(tabs);
		return tabSet;
	}

	public void undoableEditHappened(UndoableEditEvent e) {
		if (!this.doc.isHighlighting()) {
			this.editor.getUndoManager().addEdit(e.getEdit());
		}
	}

	public int getNumberOfLines() {
		return this.numberOfLines;
	}

	public void calcNumberOfLines() {

		int oldNumberOfLines = this.numberOfLines;
		this.numberOfLines = 1;
		try {
			StringBuffer text = new StringBuffer(this.doc.getText(0, this.doc
					.getLength()));
			for (int i = 0; i < text.length(); i++) {
				if (text.charAt(i) == '\n') {
					this.numberOfLines++;
				}
			}
			if (oldNumberOfLines != this.numberOfLines) {
				this.editor.getLineNumberPanel().setNumberOfLines(
						this.numberOfLines);
			}
		} catch (BadLocationException e) {
			e.printStackTrace();
			this.numberOfLines = oldNumberOfLines;
		}
	}

	public void markLine(long line) {
		this.line = -1;
		Element root = this.getDocument().getDefaultRootElement();
		if (root != null) {
			Element elem = root.getElement((int) line);
			if (elem != null) {
				this.setCaretPosition(elem.getStartOffset());
				this.line = line;
			} else {
				elem = root.getElement(root.getElementCount() - 1);
				if (elem != null) {
					this.setCaretPosition(elem.getStartOffset());
					this.line = root.getElementCount() - 1;
				}
			}
		}
		if (EventQueue.isDispatchThread()) {
			this.repaint();
		} else {
			try {
				SwingUtilities.invokeAndWait(new Runnable() {
					public void run() {
						repaint();
					}
				});
			} catch (Throwable e) {
				e.printStackTrace();
			}
		}
	}

	public boolean lineHighlightSet() {
		return this.line != -1;
	}

	public void removeLineHighlight() {
		this.line = -1;
		if (EventQueue.isDispatchThread()) {
			this.repaint();
		} else {
			try {
				SwingUtilities.invokeAndWait(new Runnable() {
					public void run() {
						repaint();
					}
				});
			} catch (Throwable e) {
				// e.printStackTrace();
			}
		}
	}

	public void locking(boolean locked) {
		if (locked) {
			this.setEditable(false);
			this.setBackground(new Color(240, 240, 240)); // getBackground());
		} else {
			this.setEditable(true);
			this.removeLineHighlight();
			this.setBackground(Color.WHITE);
		}
	}

	public void paintComponent(Graphics g) {
		if (this.line != -1) {
			int y = 0;
			try {
				Element root = this.getDocument().getDefaultRootElement();
				y = (int) this.modelToView(
						root.getElement((int) this.line).getStartOffset())
						.getY();
				g.setColor(this.getBackground());
				g.fillRect(0, 0, this.getWidth(), this.getHeight());
				g.setColor(this.getSelectionColor());
				g.fillRect(0, y, this.getWidth(), doc.lineHeight);
				this.setOpaque(false);
				super.paintComponent(g);
				this.setOpaque(true);
				return;
			} catch (Throwable e) {
			}
		}
		super.paintComponent(g);
	}

}

class InputFilter extends DocumentFilter {
	private EditorPanel editorPanel;

	InputFilter(EditorPanel p) {
		this.editorPanel = p;
	}

	public void remove(DocumentFilter.FilterBypass fb, int offset, int length)
			throws BadLocationException {
		super.remove(fb, offset, length);
		this.editorPanel.editor.setChanged(true);
		this.editorPanel.calcNumberOfLines();
	}

	public void insertString(DocumentFilter.FilterBypass fb, int offset,
			String string, AttributeSet attr) throws BadLocationException {
		super.insertString(fb, offset, string, attr);
		this.editorPanel.editor.setChanged(true);
		this.editorPanel.calcNumberOfLines();
	}

	public void replace(FilterBypass fb, int offs, int length, String str,
			AttributeSet a) throws BadLocationException {
		if (str != null && str.charAt(str.length() - 1) == KeyEvent.VK_ENTER
				&& this.editorPanel.editor.isIndent()) {
			String blanks = this.genBlanks(this.editorPanel.getDocument()
					.getText(0, this.editorPanel.getCaretPosition()),
					this.editorPanel.getCaretPosition() - 1);
			str = str + blanks;
		}

		super.replace(fb, offs, length, str, a);
		this.editorPanel.editor.setChanged(true);
		this.editorPanel.calcNumberOfLines();
	}

	private String genBlanks(String text, int pos) {
		int retIndex = -1;
		int i = pos;
		for (; i >= 0; i--) {
			if (text.charAt(i) == '\n') {
				retIndex = i;
				break;
			}
		}
		if (i != -1 && retIndex == -1) {
			return "";
		}
		String blanks = "";
		i = retIndex + 1;
		while (i <= pos) {
			if (text.charAt(i) == ' ') {
				blanks += " ";
			} else if (text.charAt(i) == '\t') {
				blanks += "\t";
			} else {
				break;
			}
			i++;
		}
		return blanks;
	}

}
