package util;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.PrintStream;
import java.util.Locale;
import java.util.Properties;

/**
 * Verwaltung der Properties
 * 
 * @author Dietrich Boles, Uni Oldenburg
 * @version 2.0 (04.08.2010)
 * 
 */
public class PropertyManager {

	public static String PROP_FILE = "solist.properties";

	public static String PROP_LANGUAGE = "language";
	public static String PROP_JDK = "jdk";
	public static String PROP_DEBUGGING = "debugging";
	public static String PROP_SCRATCH = "scratch";
	public static String PROP_INITHEAPSPACE = "initheapspace";
	public static String PROP_MAXHEAPSPACE = "maxheapspace";
	public static String PROP_HANDBOOK_ONLINE = "handbookonline";
	public static String PROP_HANDBOOK_OFFLINE = "handbookoffline";

	private static String DEF_LANGUAGE = "de"; // de / en
	private static String DEF_JDK = null;
	private static boolean DEF_DEBUGGING = true; // on / off
	private static boolean DEF_SCRATCH = false; // true / false
	private static String DEF_INITHEAPSPACE = "32"; // -Xms32m, Standard ist 2M,
	// min 1M
	private static String DEF_MAXHEAPSPACE = "128"; // -Xmx128m, Standard ist
	// 64M, min 2M

	private static String DEF_HANDBOOK_ONLINE = null; // URL
	private static String DEF_HANDBOOK_OFFLINE = null; // Dateiname

	// siehe http://java.sun.com/j2se/1.5.0/docs/tooldocs/solaris/java.html

	private static PropertyManager manager = null;
	private Properties props;

	/**
	 * erzeugt einen Property-Manager. Geladen werden die Properties aus der
	 * Datei "solist.properties"
	 */
	protected PropertyManager() {
		FileInputStream propInFile = null;
		try {
			this.props = new Properties();
			if (new File(PropertyManager.PROP_FILE).exists()) {
				propInFile = new FileInputStream(PropertyManager.PROP_FILE);
				this.props.load(propInFile);
			}
		} catch (Throwable exc) {
			exc.printStackTrace();
		} finally {
			if (propInFile != null) {
				try {
					propInFile.close();
				} catch (IOException e) {
				}
			}
		}
	}

	/**
	 * Liefert den PropertyManager
	 * 
	 * @return
	 */
	public static PropertyManager getPropertyManager() {
		if (PropertyManager.manager == null) {
			PropertyManager.manager = new PropertyManager();
		}
		return PropertyManager.manager;
	}

	/**
	 * Liefert die eingestellte Sprache (deutsch oder englisch)
	 * 
	 * @return
	 */
	public String getLanguage() {
		String lang = this.props.getProperty(PROP_LANGUAGE);
		return lang != null ? lang.trim() : PropertyManager.DEF_LANGUAGE;
	}

	/**
	 * Liefert die Locale in Abhngigkeit der eingestellten Sprache
	 * 
	 * @return
	 */
	public Locale getLocale() {
		String language = this.getLanguage();
		if (language.equals("en")) {
			return Locale.ENGLISH;
		} else if (language.equals("de")) {
			return Locale.GERMAN;
		}
		return Locale.getDefault();
	}

	/**
	 * Speichert die Property jdk. Hierin wird gespeichert, welches JDK benutzt
	 * werden soll.
	 * 
	 * @param jdk
	 */
	public void setJDK(String jdk) {
		this.props.setProperty(PROP_JDK, jdk);
		this.store();
	}

	/**
	 * Liefert die Property jdk. Hierin ist gespeichert, welches JDK benutzt
	 * werden soll. Es kann auch null geliefert werden, wenn die Property noch
	 * nicht gesetzt wurde.
	 * 
	 * @return
	 */
	public String getJDK() {
		String jdk = this.props.getProperty(PROP_JDK);
		if (jdk == null || jdk.trim().equals("")) {
			return DEF_JDK;
		} else {
			return jdk.trim();
		}
	}

	public String getInitHeapSpaceP() {
		String space = this.props.getProperty(PROP_INITHEAPSPACE);
		if (space == null) {
			return DEF_INITHEAPSPACE;
		} else {
			return space.trim();
		}
	}

	public String getMaxHeapSpaceP() {
		String space = this.props.getProperty(PROP_MAXHEAPSPACE);
		if (space == null) {
			return DEF_MAXHEAPSPACE;
		} else {
			return space.trim();
		}
	}

	public String getInitHeapSpace() {
		String space = this.props.getProperty(PROP_INITHEAPSPACE);
		if (space == null) {
			return "-Xms" + DEF_INITHEAPSPACE + "m";
		} else {
			return "-Xms" + space.trim() + "m";
		}
	}

	public String getMaxHeapSpace() {
		String space = this.props.getProperty(PROP_MAXHEAPSPACE);
		if (space == null) {
			return "-Xmx" + DEF_MAXHEAPSPACE + "m";
		} else {
			return "-Xmx" + space.trim() + "m";
		}
	}

	public String getHandbookOnline() {
		String url = this.props.getProperty(PROP_HANDBOOK_ONLINE);
		if (url == null || url.trim().equals("")) {
			return DEF_HANDBOOK_ONLINE;
		} else {
			return url.trim();
		}
	}

	public String getHandbookOffline() {
		String url = this.props.getProperty(PROP_HANDBOOK_OFFLINE);
		if (url == null || url.trim().equals("")) {
			return DEF_HANDBOOK_OFFLINE;
		} else {
			return url.trim();
		}
	}

	public boolean isDebuggingOn() {
		String value = this.props.getProperty(PROP_DEBUGGING);
		if (value == null) {
			return DEF_DEBUGGING;
		} else {
			return value.trim().equalsIgnoreCase("on");
		}
	}

	public boolean isScratch() {
		String value = this.props.getProperty(PROP_SCRATCH);
		if (value == null) {
			return DEF_SCRATCH;
		} else {
			return value.trim().equalsIgnoreCase("true");
		}
	}

	public void store() {
		this.props.list(System.out);
		FileOutputStream prop = null;
		try {
			prop = new FileOutputStream(PropertyManager.PROP_FILE);
			this.props.store(prop, "");
		} catch (Throwable th) {
			th.printStackTrace();
		} finally {
			if (prop != null) {
				try {
					prop.close();
				} catch (IOException e) {
				}
			}
		}
	}

	// Scratch Props
	private static String PROP_NEW_PROCEDURE = "scratch_new_procedure";
	private static String PROP_NEW_FUNCTION = "scratch_new_function";
	private static String PROP_VOID_RETURN = "scratch_void_return";
	private static String PROP_BOOL_RETURN = "scratch_bool_return";
	private static String PROP_TRUE = "scratch_true";
	private static String PROP_FALSE = "scratch_false";
	private static String PROP_AND = "scratch_and";
	private static String PROP_OR = "scratch_or";
	private static String PROP_NOT = "scratch_not";
	private static String PROP_IF = "scratch_if";
	private static String PROP_ELSE_IF = "scratch_else_if";
	private static String PROP_ELSE = "scratch_else";
	private static String PROP_WHILE = "scratch_while";
	private static String PROP_DO = "scratch_do";
	private static String PROP_DO_WHILE = "scratch_do_while";

	public static String DEF_NEW_PROCEDURE = "Neue Prozedur";
	public static String DEF_NEW_FUNCTION = "Neue Funktion";
	public static String DEF_VOID_RETURN = "verlasse";
	public static String DEF_BOOL_RETURN = "liefere";
	public static String DEF_TRUE = "wahr";
	public static String DEF_FALSE = "falsch";
	public static String DEF_AND = "und";
	public static String DEF_OR = "oder";
	public static String DEF_NOT = "nicht";
	public static String DEF_IF = "falls";
	public static String DEF_ELSE_IF = "falls";
	public static String DEF_ELSE = "sonst";
	public static String DEF_WHILE = "solange";
	public static String DEF_DO = "wiederhole";
	public static String DEF_DO_WHILE = "solange";

	public static String DEF_E_NEW_PROCEDURE = "Neue Prozedur";
	public static String DEF_E_NEW_FUNCTION = "Neue Funktion";
	public static String DEF_E_VOID_RETURN = "return";
	public static String DEF_E_BOOL_RETURN = "return";
	public static String DEF_E_TRUE = "true";
	public static String DEF_E_FALSE = "false";
	public static String DEF_E_AND = "and";
	public static String DEF_E_OR = "or";
	public static String DEF_E_NOT = "not";
	public static String DEF_E_IF = "if";
	public static String DEF_E_ELSE_IF = "if";
	public static String DEF_E_ELSE = "else";
	public static String DEF_E_WHILE = "while";
	public static String DEF_E_DO = "do";
	public static String DEF_E_DO_WHILE = "while";

	public void printScratchDEProps(PrintStream out) {
		out.println(PROP_NEW_PROCEDURE + "=" + DEF_NEW_PROCEDURE);
		out.println(PROP_NEW_FUNCTION + "=" + DEF_NEW_FUNCTION);
		out.println(PROP_VOID_RETURN + "=" + DEF_VOID_RETURN);
		out.println(PROP_BOOL_RETURN + "=" + DEF_BOOL_RETURN);
		out.println(PROP_TRUE + "=" + DEF_TRUE);
		out.println(PROP_FALSE + "=" + DEF_FALSE);
		out.println(PROP_AND + "=" + DEF_AND);
		out.println(PROP_OR + "=" + DEF_OR);
		out.println(PROP_NOT + "=" + DEF_NOT);
		out.println(PROP_IF + "=" + DEF_IF);
		out.println(PROP_ELSE_IF + "=" + DEF_ELSE_IF);
		out.println(PROP_ELSE + "=" + DEF_ELSE);
		out.println(PROP_WHILE + "=" + DEF_WHILE);
		out.println(PROP_DO + "=" + DEF_DO);
		out.println(PROP_DO_WHILE + "=" + DEF_DO_WHILE);
	}
	
	public void printScratchENProps(PrintStream out) {
		out.println(PROP_NEW_PROCEDURE + "=" + DEF_E_NEW_PROCEDURE);
		out.println(PROP_NEW_FUNCTION + "=" + DEF_E_NEW_FUNCTION);
		out.println(PROP_VOID_RETURN + "=" + DEF_E_VOID_RETURN);
		out.println(PROP_BOOL_RETURN + "=" + DEF_E_BOOL_RETURN);
		out.println(PROP_TRUE + "=" + DEF_E_TRUE);
		out.println(PROP_FALSE + "=" + DEF_E_FALSE);
		out.println(PROP_AND + "=" + DEF_E_AND);
		out.println(PROP_OR + "=" + DEF_E_OR);
		out.println(PROP_NOT + "=" + DEF_E_NOT);
		out.println(PROP_IF + "=" + DEF_E_IF);
		out.println(PROP_ELSE_IF + "=" + DEF_E_ELSE_IF);
		out.println(PROP_ELSE + "=" + DEF_E_ELSE);
		out.println(PROP_WHILE + "=" + DEF_E_WHILE);
		out.println(PROP_DO + "=" + DEF_E_DO);
		out.println(PROP_DO_WHILE + "=" + DEF_E_DO_WHILE);
	}

	public String getNewProcedure() {
		String prop = this.props.getProperty(PROP_NEW_PROCEDURE);
		if (prop == null || prop.trim().equals("")) {
			return DEF_NEW_PROCEDURE;
		} else {
			return prop.trim();
		}
	}

	public String getNewFunction() {
		String prop = this.props.getProperty(PROP_NEW_FUNCTION);
		if (prop == null || prop.trim().equals("")) {
			return DEF_NEW_FUNCTION;
		} else {
			return prop.trim();
		}
	}

	public String getVoidReturn() {
		String prop = this.props.getProperty(PROP_VOID_RETURN);
		if (prop == null || prop.trim().equals("")) {
			return DEF_VOID_RETURN;
		} else {
			return prop.trim();
		}
	}

	public String getBoolReturn() {
		String prop = this.props.getProperty(PROP_BOOL_RETURN);
		if (prop == null || prop.trim().equals("")) {
			return DEF_BOOL_RETURN;
		} else {
			return prop.trim();
		}
	}

	public String getAnd() {
		String prop = this.props.getProperty(PROP_AND);
		if (prop == null || prop.trim().equals("")) {
			return DEF_AND;
		} else {
			return prop.trim();
		}
	}

	public String getOr() {
		String prop = this.props.getProperty(PROP_OR);
		if (prop == null || prop.trim().equals("")) {
			return DEF_OR;
		} else {
			return prop.trim();
		}
	}

	public String getNot() {
		String prop = this.props.getProperty(PROP_NOT);
		if (prop == null || prop.trim().equals("")) {
			return DEF_NOT;
		} else {
			return prop.trim();
		}
	}

	public String getTrue() {
		String prop = this.props.getProperty(PROP_TRUE);
		if (prop == null || prop.trim().equals("")) {
			return DEF_TRUE;
		} else {
			return prop.trim();
		}
	}

	public String getFalse() {
		String prop = this.props.getProperty(PROP_FALSE);
		if (prop == null || prop.trim().equals("")) {
			return DEF_FALSE;
		} else {
			return prop.trim();
		}
	}

	public String getWhile() {
		String prop = this.props.getProperty(PROP_WHILE);
		if (prop == null || prop.trim().equals("")) {
			return DEF_WHILE;
		} else {
			return prop.trim();
		}
	}

	public String getDo() {
		String prop = this.props.getProperty(PROP_DO);
		if (prop == null || prop.trim().equals("")) {
			return DEF_DO;
		} else {
			return prop.trim();
		}
	}

	public String getDoWhile() {
		String prop = this.props.getProperty(PROP_DO_WHILE);
		if (prop == null || prop.trim().equals("")) {
			return DEF_DO_WHILE;
		} else {
			return prop.trim();
		}
	}

	public String getIf() {
		String prop = this.props.getProperty(PROP_IF);
		if (prop == null || prop.trim().equals("")) {
			return DEF_IF;
		} else {
			return prop.trim();
		}
	}

	public String getElseIf() {
		String prop = this.props.getProperty(PROP_ELSE_IF);
		if (prop == null || prop.trim().equals("")) {
			return DEF_ELSE_IF;
		} else {
			return prop.trim();
		}
	}

	public String getElse() {
		String prop = this.props.getProperty(PROP_ELSE);
		if (prop == null || prop.trim().equals("")) {
			return DEF_ELSE;
		} else {
			return prop.trim();
		}
	}

}
