import theater.*;

import java.util.ArrayList;
import java.util.List;


public class Rocket extends Actor {
    /** The maximum value that the direction can have. */
    private final static double MAX_ANGLE = 360;

    /** "rocket" is the name of the rocket */
    protected Solist rocket;

    /** The direction the rocket is facing. */
    private double direction;

    /** speed of the rocket */
    private double speed;

    /** The x location. */
    private double x;

    /** The y location. */
    private double y;
    private Stage stage;
    private TheaterImage activeImage;
    private TheaterImage notActiveImage;

    // Default-Konstruktor (muss implementiert werden!)
    public Rocket() {
        super(); // nicht loeschen!
        this.activeImage = new TheaterImage("rocketwiththrust.gif");
        this.notActiveImage = new TheaterImage("rocket.gif");
        setImage(notActiveImage);

        rocket = (Solist) this;

        this.direction = 0;
        this.speed = 5;
    }

    // Copy-Konstruktor (muss implementiert werden!)
    public Rocket(Rocket actor) {
        super(actor); // nicht loeschen!
        setImage(actor.getImage());

        rocket = (Solist) this;

        direction = actor.direction;
        speed = actor.speed;
        x = actor.x;
        y = actor.y;
        this.activeImage = actor.activeImage;
        this.notActiveImage = actor.notActiveImage;
    }

    @Override
    @Invisible
    public void addedToStage(Stage stage) {
        this.stage = stage;
        this.x = getColumn();
        this.y = getRow();
        super.setRotation((int) this.direction);
    }

    @Description("Die Rakete dreht sich um einen bestimmten Winkel nach rechts")
    public void right(double degrees) {
        Performance.getPerformance().lock();

        try {
            for (double i = 0; i <= (degrees - speed); i += speed) {
                rightIntern(speed);
            }

            rightIntern(degrees % speed);
        } finally {
            Performance.getPerformance().unlock();
        }
    }

    @Description("Die Rakete dreht sich um einen bestimmten Winkel nach links")
    public void left(double degrees) {
        Performance.getPerformance().lock();

        try {
            for (double i = 0; i <= (degrees - speed); i += speed) {
                leftIntern(speed);
            }

            leftIntern(degrees % speed);
        } finally {
            Performance.getPerformance().unlock();
        }
    }

    @Description("Die Rakete dreht sich rechtsum, " +
    "bis sie in die angegebene Richtung ausgerichtet ist")
    public void rightTo(double dir) {
        Performance.getPerformance().lock();

        try {
            dir = dir % MAX_ANGLE;

            if (dir < 0) {
                dir = MAX_ANGLE - dir;
            }

            double degrees = dir - direction;

            if (degrees < 0) {
                degrees = MAX_ANGLE + degrees;
            }

            for (double i = 0; i <= (degrees - speed); i += speed) {
                rightIntern(speed);
            }

            rightIntern(degrees % speed);
        } finally {
            Performance.getPerformance().unlock();
        }
    }

    @Description("Die Rakete dreht sich linksum, " +
    "bis sie in die angegebene Richtung ausgerichtet ist")
    public void leftTo(double dir) {
        Performance.getPerformance().lock();

        try {
            dir = dir % MAX_ANGLE;

            if (dir < 0) {
                dir = MAX_ANGLE - dir;
            }

            double degrees = direction - dir;

            if (degrees < 0) {
                degrees = MAX_ANGLE + degrees;
            }

            for (double i = 0; i <= (degrees - speed); i += speed) {
                leftIntern(speed);
            }

            leftIntern(degrees % speed);
        } finally {
            Performance.getPerformance().unlock();
        }
    }

    @Description("Die Rakete dreht sich, je nachdem, was schneller ist, " +
    "linksum oder rechtsum, " +
    "bis sie in die angegebene Richtung ausgerichtet ist")
    public void turnTo(double dir) {
        Performance.getPerformance().lock();

        try {
            dir = dir % MAX_ANGLE;

            if (dir < 0) {
                dir = MAX_ANGLE - dir;
            }

            if (isTurnLeft(dir)) {
                leftTo(dir);
            } else {
                rightTo(dir);
            }
        } finally {
            Performance.getPerformance().unlock();
        }
    }

    @Description("Die Rakete dreht sich, je nachdem, was schneller ist, " +
    "linksum oder rechtsum, " +
    "bis sie in Richtung des angegebenen Artefaktes ausgerichtet ist.")
    public void turnTo(Artifact object) {
        Performance.getPerformance().lock();

        try {
            if ((object != null) && (object.getStage() != null)) {
                turnTo(object.getColumn(), object.getRow());
            }
        } finally {
            Performance.getPerformance().unlock();
        }
    }

    @Description("Die Rakete dreht sich, je nachdem, was schneller ist, " +
    "linksum oder rechtsum, " +
    "bis sie in Richtung des angegebenen Punktes ausgerichtet ist.")
    public void turnTo(int toX, int toY) {
        Performance.getPerformance().lock();

        try {
            turnTo(calcDirection(toX, toY));
        } finally {
            Performance.getPerformance().unlock();
        }
    }

    @Description("Die Rakete bewegt sich um eine bestimmte Anzahl von Einheiten nach vorne")
    public void move(double distance) {
        if (distance <= 0) {
            return;
        }

        Performance.getPerformance().lock();

        try {
            for (double i = 0; i <= (distance - speed); i += speed) {
                forwardIntern(speed);
            }

            forwardIntern(distance % speed);
        } finally {
            Performance.getPerformance().unlock();
        }
    }

    @Description("Die Rakete fliegt zum angegebenen Punkt")
    public void moveTo(int toX, int toY) {
        Performance.getPerformance().lock();

        try {
            double sum = 0;
            double step = speed;

            while (!reached(toX, toY)) {
                double dir = calcDirection(toX, toY);

                if (isTurnLeft(dir)) {
                    leftIntern(step);
                    sum -= step;
                } else {
                    rightIntern(step);
                    sum += step;
                }

                if (Math.abs(sum) > 360) {
                    sum = 0;
                    step = 2 * step;
                }

                forwardIntern(speed);
            }

            moveToIntern(toX, toY);
        } finally {
            Performance.getPerformance().unlock();
        }
    }

    @Description("Die Rakete fliegt zum angegebenen Artefakt")
    public void moveTo(Artifact object) {
        Performance.getPerformance().lock();

        try {
            if ((object != null) && (object.getStage() != null)) {
                moveTo(object.getColumn(), object.getRow());
            }
        } finally {
            Performance.getPerformance().unlock();
        }
    }

    @Description("Die Rakete feuert einen Schuss ab")
    public void fire() {
        Performance.getPerformance().lock();

        try {
            beforeAction();

            Bullet b = new Bullet(this);
            stage.add(b, getColumn(), getRow());
            afterAction();
        } finally {
            Performance.getPerformance().unlock();
        }
    }

    @Description("Liefert ein sich im Weltall befindendes Artefakt der angegebenen Klasse")
    public Artifact find(Class<?extends Artifact> cls) {
        Performance.getPerformance().lock();

        try {
            List<Component> objects = stage.getComponents(cls);

            if ((objects != null) && (objects.size() > 0)) {
                int index = (new java.util.Random()).nextInt(objects.size());

                return (Artifact) objects.get(index);
            } else {
                return null;
            }
        } finally {
            Performance.getPerformance().unlock();
        }
    }

    @Description("Liefert alle sich im Weltall befindenden Artefakte der angegebenen Klasse")
    public List<Artifact> findAll(Class<?extends Artifact> cls) {
        Performance.getPerformance().lock();

        try {
            ArrayList<Artifact> props = new ArrayList<Artifact>();
            List<Component> objects = stage.getComponents(cls);

            if (objects != null) {
                for (Component comp : objects) {
                    props.add((Artifact) comp);
                }
            }

            return props;
        } finally {
            Performance.getPerformance().unlock();
        }
    }

    @Description("Liefert die Entfernung der Rakete zum angegebenen Punkt")
    public double getDistance(int toX, int toY) {
        Performance.getPerformance().lock();

        try {
            int fromX = getColumn();
            int fromY = getRow();

            return Math.sqrt(Math.pow(toX - fromX, 2.0) +
                Math.pow(toY - fromY, 2.0));
        } finally {
            Performance.getPerformance().unlock();
        }
    }

    @Description("Liefert die Entfernung der Rakete zum angegebenen Artefakt")
    public double getDistance(Prop object) {
        Performance.getPerformance().lock();

        try {
            if ((object != null) && (object.getStage() != null)) {
                return getDistance(object.getColumn(), object.getRow());
            } else {
                return -1.0;
            }
        } finally {
            Performance.getPerformance().unlock();
        }
    }

    @Description("Liefert die x-Koordinate der Rakete")
    public int getX() {
        Performance.getPerformance().lock();

        try {
            return getColumn();
        } finally {
            Performance.getPerformance().unlock();
        }
    }

    @Description("Liefert die y-Koordinate der Rakete")
    public int getY() {
        Performance.getPerformance().lock();

        try {
            return getRow();
        } finally {
            Performance.getPerformance().unlock();
        }
    }

    @Description("Liefert die Weite des Weltalls")
    public int getWidthOfSpace() {
        Performance.getPerformance().lock();

        try {
            return stage.getNumberOfColumns();
        } finally {
            Performance.getPerformance().unlock();
        }
    }

    @Description("Liefert die Hhe des Weltalls")
    public int getHeightOfSpace() {
        Performance.getPerformance().lock();

        try {
            return stage.getNumberOfRows();
        } finally {
            Performance.getPerformance().unlock();
        }
    }

    @Description("Liefert einen Vector, mittels dem Flugrichtung und Geschwindigkeit " +
    "der Rakete ermittelt werden kann")
    public Vector getMovementVector() {
        Performance.getPerformance().lock();

        try {
            return new Vector(getRotation(), speed);
        } finally {
            Performance.getPerformance().unlock();
        }
    }

    @Description("ndert die Geschwindigkeit der Rakete")
    public void changeSpeed(double newSpeed) {
        Performance.getPerformance().lock();

        try {
            beforeAction();
            speed = newSpeed;
            afterAction();
        } finally {
            Performance.getPerformance().unlock();
        }
    }

    //*************************************
    // internal and invisble methods
    //*************************************

    // Set the location of the rocket.
    @Invisible
    private final void setLocation(double x, double y) {
        this.x = x;
        this.y = y;
        super.setLocation((int) Math.floor(x), (int) Math.floor(y));
    }

    // We need to override this method, so we can interactively move objects.
    // This method should not be used by subclasses. Use the 
    // setLocation(double x, double y) instead
    @Override
    @Invisible
    public final void setLocation(int x, int y) {
        this.x = x;
        this.y = y;
        super.setLocation(x, y);
    }

    @Invisible
    private void rightIntern(double degrees) {
        if (degrees < 0) {
            left(-degrees);
        } else {
            direction = direction + degrees;

            if (direction > MAX_ANGLE) {
                direction = direction % MAX_ANGLE;
            }

            setRotation((int) Math.round(direction));
        }
    }

    @Invisible
    private void leftIntern(double degrees) {
        if (degrees < 0) {
            right(-degrees);
        } else {
            degrees = degrees % MAX_ANGLE;
            direction = direction - degrees;

            if (direction < 0) {
                direction = MAX_ANGLE + direction;
            }

            setRotation((int) Math.round(direction));
        }
    }

    @Invisible
    private boolean isTurnLeft(double dir) {
        dir = dir % MAX_ANGLE;

        if (dir < 0) {
            dir = MAX_ANGLE - dir;
        }

        double links = 0;

        if ((direction - dir) >= 0) {
            links = direction - dir;
        } else {
            links = (MAX_ANGLE + direction) - dir;
        }

        double rechts = 0;

        if ((dir - direction) >= 0) {
            rechts = dir - direction;
        } else {
            rechts = (MAX_ANGLE + dir) - direction;
        }

        return (links <= rechts);
    }

    @Invisible
    private void forwardIntern(double distance) {
        if (distance < 0) {
            back(-distance);
        } else {
            Vector speed = new Vector(getRotation(), distance);
            moveToIntern(x + speed.getX(), y + speed.getY());
        }
    }

    @Invisible
    private void backIntern(double distance) {
        if (distance < 0) {
            move(-distance);
        } else {
            Vector speed = new Vector(getRotation(), distance);
            moveToIntern(x - speed.getX(), y - speed.getY());
        }
    }

    @Invisible
    private double calcDirection(int toX, int toY) {
        int fromX = getColumn();
        int fromY = getRow();

        double dx = Math.abs(toX - fromX);
        double dy = Math.abs(toY - fromY);
        double alpha = 0;
        double dir = 0;

        if ((fromX == toX) && (fromY == toY)) {
            dir = 0;
        } else if (fromX <= toX) {
            if (fromY < toY) {
                // rechts unten
                alpha = Math.toDegrees(Math.atan(dx / dy));
                dir = 180 - alpha;
            } else {
                // rechts oben
                alpha = Math.toDegrees(Math.atan(dy / dx));
                dir = 90 - alpha;
            }
        } else {
            if (fromY <= toY) {
                // links unten
                alpha = Math.toDegrees(Math.atan(dx / dy));
                dir = MAX_ANGLE - (180 - alpha);
            } else {
                // links oben
                alpha = Math.toDegrees(Math.atan(dy / dx));
                dir = MAX_ANGLE - (90 - alpha);
            }
        }

        return dir;
    }

    @Invisible
    private boolean reached(int toX, int toY) {
        int x = getColumn();
        int y = getRow();

        return (Math.abs(x - toX) <= 2) && (Math.abs(y - toY) <= 2);
    }

    // Moves the rocket to the given position.
    @Invisible
    private void moveToIntern(double newX, double newY) {
        Stage stage = getStage();

        if ((newX >= 0) && (newY >= 0) && (newX < stage.getNumberOfColumns()) &&
                (newY < stage.getNumberOfRows())) {
            beforeAction();
            setLocation(newX, newY);
            checkCollisions();
            afterAction();
        }
    }

    @Invisible
    private double correctDirection() {
        double dir = direction - 90.0;

        if (dir < 0) {
            dir = MAX_ANGLE + dir;
        }

        return dir;
    }

    @Description("Die Rakete bewegt sich um eine bestimmte Anzahl von Einheiten nach hinten")
    @Invisible
    private void back(double distance) {
        Performance.getPerformance().lock();

        try {
            for (double i = 0; i <= (distance - speed); i += speed) {
                backIntern(speed);
            }

            backIntern(distance % speed);
        } finally {
            Performance.getPerformance().unlock();
        }
    }

    @Description("Die Rakete fliegt zum angegebenen Punkt")
    @Invisible
    private void flyTo2(int toX, int toY) {
        Performance.getPerformance().lock();

        try {
            while (!reached(toX, toY)) {
                turnTo(toX, toY);
                forwardIntern(speed);
            }

            moveToIntern(toX, toY);
        } finally {
            Performance.getPerformance().unlock();
        }
    }

    @Override
    @Invisible
    public void setRotation(int dir) {
        beforeAction();
        super.setRotation(dir);
        afterAction();
    }

    @Invisible
    private void checkCollisions() {
        List<Component> comps = getStage()
                                    .getIntersectingComponents(this,
                Asteroid.class);

        if (comps.size() > 0) {
            Explosion ex = new Explosion();
            stage.add(ex, getColumn(), getRow());

            ((Space) stage).death();
            setFinished();
            throw new RuntimeException("tot");
        }
    }

    @Invisible
    void setFinished() {
        setImage(notActiveImage);
    }

    @Invisible
    private void beforeAction() {
        if (getImage() != activeImage) {
            setImage(activeImage);
        }
    }

    @Invisible
    private void afterAction() {
        ((Space) stage).performArtifacts();
    }
}
