package editor;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.HierarchyEvent;
import java.awt.event.HierarchyListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.util.Observer;

import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.ButtonGroup;
import javax.swing.JButton;
import javax.swing.JCheckBoxMenuItem;
import javax.swing.JFrame;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JRadioButtonMenuItem;
import javax.swing.JScrollPane;
import javax.swing.JSplitPane;
import javax.swing.JToolBar;
import javax.swing.KeyStroke;
import javax.swing.WindowConstants;
import javax.swing.text.DefaultEditorKit;
import javax.swing.undo.UndoManager;

import model.Play;
import util.ResourceLoader;
import util.ResourceManager;
import view.MessagePanel;

import compiler.CompilerErrorPanel;

import editor.listener.CloseEditorListener;
import editor.listener.FontSizeListener;
import editor.listener.FormatListener;
import editor.listener.IndentListener;
import editor.listener.PrintListener;
import editor.listener.RedoListener;
import editor.listener.SaveFileListener;
import editor.listener.UndoListener;

/**
 * Realisiert einen Editor zum Editieren von Dateien.
 * 
 * Ist gem des Observer/Observable-Pattern implementiert. Werden nderungen im
 * Editor vorgenommen (neue Eingabe, abspeichern) werden die Observer darber
 * informiert.
 * 
 * @author Dietrich Boles, Uni Oldenburg
 * @version 1.0 (12.11.2008)
 * 
 */
public class Editor extends JFrame {

	String fileName;

	JMenuBar menuBar;

	JMenu fileMenu;
	JMenuItem saveMenuItem;
	JMenuItem printMenuItem;
	JMenuItem closeMenuItem;

	JMenu editMenu;
	JMenuItem undoMenuItem;
	JMenuItem redoMenuItem;
	JMenuItem cutMenuItem;
	JMenuItem copyMenuItem;
	JMenuItem pasteMenuItem;

	JMenu formatMenu;
	JCheckBoxMenuItem indentMenuItem;
	JMenuItem formatMenuItem;
	JMenu fontsizeMenu;
	JRadioButtonMenuItem size8, size10, size12, size14, size16, size18, size24,
			size36, size48;

	JToolBar toolBar;
	JButton saveButton;
	JButton printButton;
	JButton cutButton;
	JButton copyButton;
	JButton pasteButton;
	JButton undoButton;
	JButton redoButton;

	JPanel mainPanel;
	EditorPanel editorPanel;
	JPanel editorPanelSurrounding;
	LineNumberPanel lineNumbePanel;
	CompilerErrorPanel errorPanel;

	MessagePanel messagePanel;

	ResourceManager resourceManager;

	boolean changed;
	boolean saved;
	boolean isJava;
	boolean indent;

	int fontSize = 12;

	UndoManager undoManager;

	EditorObservable editorObservable;

	/**
	 * Erzeugt einen neuen Editor; bergeben wird der Name der zu verwaltenen
	 * Datei sowie ein weiterer (bergeordneter) Name
	 * 
	 * @param fileName
	 * @param playName
	 * @param isJava
	 */
	public Editor(String fileName, String playName, boolean isJava) {
		super();
		this.isJava = isJava;
		this.setDefaultCloseOperation(WindowConstants.DO_NOTHING_ON_CLOSE);
		this.addWindowListener(new WindowAdapter() {
			public void windowClosing(WindowEvent e) {
				new CloseEditorListener(Editor.this).closeEditor();
			}
		});
		this.editorObservable = new EditorObservable(this);
		this.changed = false;
		this.saved = true;
		this.indent = true;
		this.resourceManager = ResourceManager.getResourceManager();
		this.fileName = fileName;
		File f = new File(fileName);
		if (playName == null || playName.equals("")) {
			this.setTitle(f.getName());
		} else {
			this.setTitle(playName + ": " + f.getName());
		}
		this.setFocusable(true);
		this.setLayout(new BorderLayout());

		this.createComponents();
		this.initComponents();
		this.pack();
	}

	/**
	 * Anmeldung eines Observers. Dieser wird jeweils benachrichtigt, wenn sich
	 * der Zustand des Editors gendert hat. Inwieweit er sich gendert hat,
	 * kann ber die Methode hasChanged abgefragt werden.
	 * 
	 * @param observer
	 */
	public void addObserver(Observer observer) {
		this.editorObservable.addObserver(observer);
	}

	/**
	 * Abfrage, ob der Inhalt gendert aber noch nicht gespeichert wurde.
	 * 
	 * @return
	 */
	public boolean hasChanged() {
		return this.changed;
	}

	/**
	 * Speichert den Inhalt des Editors in der verbundenen Datei ab.
	 * 
	 * @throws IOException
	 */
	public void save() throws IOException {
		File file = new File(this.fileName);
		FileWriter fw = null;
		try {
			fw = new FileWriter(file);
			// out = new OutputStreamWriter(new FileOutputStream(file),
			// "UTF-8");
			fw.write(this.editorPanel.getText());
			fw.flush();
		} finally {
			if (fw != null) {
				fw.close();
			}
		}
		this.setChanged(false);
		this.setSaved();
	}

	public void setSaved() {
		this.saved = true;
		this.editorObservable.setChanged();
		this.editorObservable.notifyObservers();
		this.editorObservable.clearChanged();
	}

	private boolean savedAs = false;

	public void setSavedAs() {
		this.savedAs = true;
	}

	public boolean isSavedAs() {
		return this.savedAs;
	}

	public void saveTmp(String name) throws IOException {
		File file = new File(name);
		FileWriter fw = null;
		try {
			fw = new FileWriter(file);
			// fw = new OutputStreamWriter(new FileOutputStream(file), "UTF-8");
			fw.write(this.editorPanel.getText());
			fw.flush();
		} finally {
			if (fw != null) {
				fw.close();
			}
		}
	}

	/**
	 * Liefert den Speicher-Zustand des Editors
	 * 
	 * @return
	 */
	public boolean isSaved() {
		return this.saved;
	}

	/**
	 * Schliet den Editor (wird unsichtbar gemacht)
	 */
	public void close() {
		this.setVisible(false);
	}

	/**
	 * ffnet den Editor (wird geladen und sichtbar gemacht).
	 */
	public void open() {
		this.setVisible(false);
		this.editorPanel.loadFile(this.fileName);
		this.setVisible(true);
	}

	/**
	 * Ist der Editor geffnet oder geschlossen
	 * 
	 * @return
	 */
	public boolean isOpen() {
		return this.isVisible();
	}

	/**
	 * Liefert den Dateinamen
	 * 
	 * @return
	 */
	public String getFileName() {
		return this.fileName;
	}

	public EditorPanel getEditorPanel() {
		return this.editorPanel;
	}

	public LineNumberPanel getLineNumberPanel() {
		return this.lineNumbePanel;
	}

	public MessagePanel getMessagePanel() {
		if (this != Play.getPlay().getPlayFrame().getSolistEditor().getEditor()) {
			return this.messagePanel;
		} else {
			return Play.getPlay().getPlayFrame().getMessagePanel();
		}
	}

	public UndoManager getUndoManager() {
		return this.undoManager;
	}

	public JMenuItem getUndoMenueItem() {
		return this.undoMenuItem;
	}

	public JButton getUndoButton() {
		return this.undoButton;
	}

	public JMenuItem getRedoMenueItem() {
		return this.redoMenuItem;
	}

	public JButton getRedoButton() {
		return this.redoButton;
	}

	/**
	 * Markiert den Editor als gendert oder nicht gendert
	 * 
	 * @param changed
	 */
	public void setChanged(boolean changed) {
		if (this.changed != changed) {
			this.changed = changed;
			if (this.changed) {
				this.setTitle(this.getTitle() + " *");
				this.saveMenuItem.setEnabled(true);
				this.saveButton.setEnabled(true);
				this.saved = false;
				this.editorObservable.setChanged();
				this.editorObservable.notifyObservers();
				this.editorObservable.clearChanged();
			} else {
				String t = this.getTitle();
				this.setTitle(t.substring(0, t.length() - 2));
				this.saveMenuItem.setEnabled(false);
				this.saveButton.setEnabled(false);
			}

		}
		if (changed) {
			this.savedAs = false;
		}
		this.setUndoRedoComponents();
	}

	public void setUndoRedoComponents() {
		if (this.undoManager.canUndo()) {
			this.undoMenuItem.setEnabled(true);
			this.undoButton.setEnabled(true);
		} else {
			this.undoMenuItem.setEnabled(false);
			this.undoButton.setEnabled(false);
		}
		if (this.undoManager.canRedo()) {
			this.redoMenuItem.setEnabled(true);
			this.redoButton.setEnabled(true);
		} else {
			this.redoMenuItem.setEnabled(false);
			this.redoButton.setEnabled(false);
		}
	}

	protected void createComponents() {
		this.menuBar = new JMenuBar();
		this.fileMenu = new JMenu(this.resourceManager.getValue("editor.file"));
		this.editMenu = new JMenu(this.resourceManager.getValue("editor.edit"));
		this.formatMenu = new JMenu(this.resourceManager
				.getValue("editor.format"));
		this.toolBar = new JToolBar();
		this.mainPanel = new JPanel();
		this.errorPanel = new CompilerErrorPanel(this);
		this.editorPanel = new EditorPanel(this);
		this.messagePanel = new MessagePanel();
	}

	protected void initComponents() {
		this.initMenues();
		this.initToolBar();
		this.initEditorPane();
		this.initMessagePanel();
	}

	protected void initMenues() {
		this.initFileMenu();
		this.menuBar.add(this.fileMenu);
		this.initEditMenu();
		this.menuBar.add(this.editMenu);
		this.initFormatMenu();
		this.menuBar.add(this.formatMenu);
		this.setJMenuBar(this.menuBar);
	}

	protected void initFileMenu() {
		this.saveMenuItem = new JMenuItem();
		this.saveMenuItem.setText(this.resourceManager
				.getValue("editor.save.text"));
		this.saveMenuItem.setToolTipText(this.resourceManager
				.getValue("editor.save.tooltip"));
		this.saveMenuItem.setMnemonic(this.resourceManager.getValue(
				"editor.save.mnemonic").charAt(0));
		this.saveMenuItem.setAccelerator(KeyStroke
				.getKeyStroke(this.resourceManager
						.getValue("editor.save.keystroke")));
		this.saveMenuItem.addActionListener(new SaveFileListener(this));
		this.saveMenuItem.setEnabled(false);
		this.fileMenu.add(this.saveMenuItem);

		this.printMenuItem = new JMenuItem();
		this.printMenuItem.setText(this.resourceManager
				.getValue("editor.print.text"));
		this.printMenuItem.setToolTipText(this.resourceManager
				.getValue("editor.print.tooltip"));
		this.printMenuItem.setMnemonic(this.resourceManager.getValue(
				"editor.print.mnemonic").charAt(0));
		this.printMenuItem.setAccelerator(KeyStroke
				.getKeyStroke(this.resourceManager
						.getValue("editor.print.keystroke")));
		this.printMenuItem.addActionListener(new PrintListener(this));
		this.fileMenu.add(this.printMenuItem);

		this.fileMenu.addSeparator();

		this.closeMenuItem = new JMenuItem();
		this.closeMenuItem.setText(this.resourceManager
				.getValue("editor.close.text"));
		this.closeMenuItem.setToolTipText(this.resourceManager
				.getValue("editor.close.tooltip"));
		this.closeMenuItem.setMnemonic(this.resourceManager.getValue(
				"editor.close.mnemonic").charAt(0));
		this.closeMenuItem.setAccelerator(KeyStroke
				.getKeyStroke(this.resourceManager
						.getValue("editor.close.keystroke")));
		this.closeMenuItem.addActionListener(new CloseEditorListener(this));
		this.fileMenu.add(this.closeMenuItem);
	}

	protected void initEditMenu() {
		this.undoMenuItem = new JMenuItem();
		this.undoMenuItem.setText(this.resourceManager
				.getValue("editor.undo.text"));
		this.undoMenuItem.setToolTipText(this.resourceManager
				.getValue("editor.undo.tooltip"));
		this.undoMenuItem.setMnemonic(this.resourceManager.getValue(
				"editor.undo.mnemonic").charAt(0));
		this.undoMenuItem.setAccelerator(KeyStroke
				.getKeyStroke(this.resourceManager
						.getValue("editor.undo.keystroke")));
		this.undoMenuItem.addActionListener(new UndoListener(this));
		this.editMenu.add(this.undoMenuItem);

		this.redoMenuItem = new JMenuItem();
		this.redoMenuItem.setText(this.resourceManager
				.getValue("editor.redo.text"));
		this.redoMenuItem.setToolTipText(this.resourceManager
				.getValue("editor.redo.tooltip"));
		this.redoMenuItem.setMnemonic(this.resourceManager.getValue(
				"editor.redo.mnemonic").charAt(0));
		this.redoMenuItem.setAccelerator(KeyStroke
				.getKeyStroke(this.resourceManager
						.getValue("editor.redo.keystroke")));
		this.redoMenuItem.addActionListener(new RedoListener(this));
		this.editMenu.add(this.redoMenuItem);

		this.editMenu.addSeparator();

		this.cutMenuItem = new JMenuItem(new MyCutAction(this));
		this.cutMenuItem.setText(this.resourceManager
				.getValue("editor.cut.text"));
		this.cutMenuItem.setToolTipText(this.resourceManager
				.getValue("editor.cut.tooltip"));
		this.cutMenuItem.setMnemonic(this.resourceManager.getValue(
				"editor.cut.mnemonic").charAt(0));
		this.cutMenuItem.setAccelerator(KeyStroke
				.getKeyStroke(this.resourceManager
						.getValue("editor.cut.keystroke")));
		this.editMenu.add(this.cutMenuItem);

		this.copyMenuItem = new JMenuItem(new MyCopyAction(this));
		this.copyMenuItem.setText(this.resourceManager
				.getValue("editor.copy.text"));
		this.copyMenuItem.setToolTipText(this.resourceManager
				.getValue("editor.copy.tooltip"));
		this.copyMenuItem.setMnemonic(this.resourceManager.getValue(
				"editor.copy.mnemonic").charAt(0));
		this.copyMenuItem.setAccelerator(KeyStroke
				.getKeyStroke(this.resourceManager
						.getValue("editor.copy.keystroke")));
		this.editMenu.add(this.copyMenuItem);

		this.pasteMenuItem = new JMenuItem(new MyPasteAction(this));
		this.pasteMenuItem.setText(this.resourceManager
				.getValue("editor.paste.text"));
		this.pasteMenuItem.setToolTipText(this.resourceManager
				.getValue("editor.paste.tooltip"));
		this.pasteMenuItem.setMnemonic(this.resourceManager.getValue(
				"editor.paste.mnemonic").charAt(0));
		this.pasteMenuItem.setAccelerator(KeyStroke
				.getKeyStroke(this.resourceManager
						.getValue("editor.paste.keystroke")));
		this.editMenu.add(this.pasteMenuItem);

	}

	protected void initFormatMenu() {
		this.indentMenuItem = new JCheckBoxMenuItem();
		this.indentMenuItem.setSelected(this.indent);
		this.indentMenuItem.setText(this.resourceManager
				.getValue("editor.indent.text"));
		this.indentMenuItem.setToolTipText(this.resourceManager
				.getValue("editor.indent.tooltip"));
		this.indentMenuItem.setMnemonic(this.resourceManager.getValue(
				"editor.indent.mnemonic").charAt(0));
		this.indentMenuItem.setAccelerator(KeyStroke
				.getKeyStroke(this.resourceManager
						.getValue("editor.indent.keystroke")));
		this.indentMenuItem.addActionListener(new IndentListener(this));
		this.formatMenu.add(this.indentMenuItem);

		this.fontsizeMenu = new JMenu();
		this.fontsizeMenu.setText(this.resourceManager
				.getValue("editor.font.text"));
		this.fontsizeMenu.setToolTipText(this.resourceManager
				.getValue("editor.font.tooltip"));
		this.fontsizeMenu.setMnemonic(this.resourceManager.getValue(
				"editor.font.mnemonic").charAt(0));
		this.formatMenu.add(this.fontsizeMenu);

		ButtonGroup group = new ButtonGroup();

		this.size8 = new JRadioButtonMenuItem("8");
		this.size8.addActionListener(new FontSizeListener(this, 8));
		this.fontsizeMenu.add(this.size8);
		group.add(this.size8);
		this.size10 = new JRadioButtonMenuItem("10");
		this.size10.addActionListener(new FontSizeListener(this, 10));
		this.fontsizeMenu.add(this.size10);
		group.add(this.size10);
		this.size12 = new JRadioButtonMenuItem("12");
		this.size12.addActionListener(new FontSizeListener(this, 12));
		this.fontsizeMenu.add(this.size12);
		group.add(this.size12);
		this.size14 = new JRadioButtonMenuItem("14");
		this.size14.addActionListener(new FontSizeListener(this, 14));
		this.fontsizeMenu.add(this.size14);
		group.add(this.size14);
		this.size16 = new JRadioButtonMenuItem("16");
		this.size16.addActionListener(new FontSizeListener(this, 16));
		this.fontsizeMenu.add(this.size16);
		group.add(this.size16);
		this.size18 = new JRadioButtonMenuItem("18");
		this.size18.addActionListener(new FontSizeListener(this, 18));
		this.fontsizeMenu.add(this.size18);
		group.add(this.size18);
		this.size24 = new JRadioButtonMenuItem("24");
		this.size24.addActionListener(new FontSizeListener(this, 24));
		this.fontsizeMenu.add(this.size24);
		group.add(this.size24);
		this.size36 = new JRadioButtonMenuItem("36");
		this.size36.addActionListener(new FontSizeListener(this, 36));
		this.fontsizeMenu.add(this.size36);
		group.add(this.size36);
		this.size48 = new JRadioButtonMenuItem("48");
		this.size48.addActionListener(new FontSizeListener(this, 48));
		this.fontsizeMenu.add(this.size48);
		group.add(this.size48);

		switch (this.fontSize) {
		case 8:
			this.size8.setSelected(true);
			break;
		case 10:
			this.size10.setSelected(true);
			break;
		case 12:
			this.size12.setSelected(true);
			break;
		case 14:
			this.size14.setSelected(true);
			break;
		case 16:
			this.size16.setSelected(true);
			break;
		case 18:
			this.size18.setSelected(true);
			break;
		case 24:
			this.size24.setSelected(true);
			break;
		case 36:
			this.size36.setSelected(true);
			break;
		default:
			this.size48.setSelected(true);
			break;
		}

		this.formatMenuItem = new JMenuItem();
		this.formatMenuItem.setText(this.resourceManager
				.getValue("editor.format.text"));
		this.formatMenuItem.setToolTipText(this.resourceManager
				.getValue("editor.format.tooltip"));
		this.formatMenuItem.setMnemonic(this.resourceManager.getValue(
				"editor.format.mnemonic").charAt(0));
		this.formatMenuItem.setAccelerator(KeyStroke
				.getKeyStroke(this.resourceManager
						.getValue("editor.format.keystroke")));
		this.formatMenuItem.addActionListener(new FormatListener(this));
		this.formatMenu.add(this.formatMenuItem);

	}

	protected void initToolBar() {
		this.toolBar.setBackground(new Color(255, 215, 180));

		this.saveButton = new JButton(ResourceLoader.getImageIcon("Save24.gif"));
		this.saveButton.setToolTipText(this.resourceManager
				.getValue("editor.toolbar.save"));
		this.saveButton.setMargin(new Insets(0, 0, 0, 0));
		this.saveButton.addActionListener(new SaveFileListener(this));
		this.saveButton.setEnabled(false);
		this.toolBar.add(this.saveButton);
		// this.toolBar.add(Box.createRigidArea(new Dimension(2, 2)));

		this.printButton = new JButton(ResourceLoader
				.getImageIcon("Print24.gif"));
		this.printButton.setToolTipText(this.resourceManager
				.getValue("editor.toolbar.print"));
		this.printButton.setMargin(new Insets(0, 0, 0, 0));
		this.printButton.addActionListener(new PrintListener(this));
		this.toolBar.add(this.printButton);
		this.toolBar.add(Box.createRigidArea(new Dimension(11, 11)));

		this.cutButton = new JButton(new MyCutAction(this));
		this.cutButton.setIcon(ResourceLoader.getImageIcon("Cut24.gif"));
		this.cutButton.setText("");
		this.cutButton.setToolTipText(this.resourceManager
				.getValue("editor.toolbar.cut"));
		this.cutButton.setMargin(new Insets(0, 0, 0, 0));
		this.toolBar.add(this.cutButton);
		// this.toolBar.add(Box.createRigidArea(new Dimension(2, 2)));

		this.copyButton = new JButton(new MyCopyAction(this));
		this.copyButton.setIcon(ResourceLoader.getImageIcon("Copy24.gif"));
		this.copyButton.setText("");
		this.copyButton.setToolTipText(this.resourceManager
				.getValue("editor.toolbar.copy"));
		this.copyButton.setMargin(new Insets(0, 0, 0, 0));
		this.toolBar.add(this.copyButton);
		// this.toolBar.add(Box.createRigidArea(new Dimension(2, 2)));

		this.pasteButton = new JButton(new MyPasteAction(this));
		this.pasteButton.setIcon(ResourceLoader.getImageIcon("Paste24.gif"));
		this.pasteButton.setText("");
		this.pasteButton.setToolTipText(this.resourceManager
				.getValue("editor.toolbar.paste"));
		this.pasteButton.setMargin(new Insets(0, 0, 0, 0));
		this.toolBar.add(this.pasteButton);
		this.toolBar.add(Box.createRigidArea(new Dimension(11, 11)));

		this.undoButton = new JButton(ResourceLoader.getImageIcon("Undo24.gif"));
		this.undoButton.setToolTipText(this.resourceManager
				.getValue("editor.toolbar.undo"));
		this.undoButton.setMargin(new Insets(0, 0, 0, 0));
		this.undoButton.addActionListener(new UndoListener(this));
		this.toolBar.add(this.undoButton);
		// this.toolBar.add(Box.createRigidArea(new Dimension(2, 2)));

		this.redoButton = new JButton(ResourceLoader.getImageIcon("Redo24.gif"));
		this.redoButton.setToolTipText(this.resourceManager
				.getValue("editor.toolbar.redo"));
		this.redoButton.setMargin(new Insets(0, 0, 0, 0));
		this.redoButton.addActionListener(new RedoListener(this));
		this.toolBar.add(this.redoButton);

		this.add(this.toolBar, BorderLayout.NORTH);
	}

	protected void initEditorPane() {

		this.undoManager = new UndoManager();
		this.undoManager.setLimit(-1);

		this.lineNumbePanel = new LineNumberPanel(this);

		if (!Play.getPlay().isEmpty()) {
			this.editorPanel.loadFile(this.fileName);
		}

		this.editorPanel.getDocument()
				.addUndoableEditListener(this.editorPanel);
		this.setUndoRedoComponents();

		this.editorPanelSurrounding = new JPanel();
		this.editorPanelSurrounding.setBorder(BorderFactory
				.createLineBorder(Color.DARK_GRAY));
		this.editorPanelSurrounding.setBackground(new Color(255, 255, 220));
		this.editorPanelSurrounding.setLayout(new BorderLayout());
		this.editorPanelSurrounding.add(Box.createRigidArea(new Dimension(12,
				12)), BorderLayout.NORTH);
		this.editorPanelSurrounding.add(Box.createRigidArea(new Dimension(12,
				12)), BorderLayout.SOUTH);
		this.editorPanelSurrounding.add(Box.createRigidArea(new Dimension(12,
				12)), BorderLayout.WEST);
		this.editorPanelSurrounding.add(Box.createRigidArea(new Dimension(12,
				12)), BorderLayout.EAST);

		JScrollPane scrollP = new JScrollPane(this.editorPanel);
		scrollP.setPreferredSize(new Dimension(700, 500));
		scrollP.getViewport().setBackground(Color.WHITE);
		scrollP.setRowHeaderView(this.lineNumbePanel);

		this.editorPanelSurrounding.add(scrollP, BorderLayout.CENTER);

		this.mainPanel.setLayout(new BorderLayout());
		if (this.isJava) {
			this.addErrorPanel();
		} else {
			this.mainPanel
					.add(this.editorPanelSurrounding, BorderLayout.CENTER);
		}
		this.add(this.mainPanel, BorderLayout.CENTER);

	}

	JSplitPane splitPane;
	JScrollPane scrollP;
	boolean errorPaneAdded = false;
	JPanel textPaneSurrounding;

	public void addErrorPanel() {
		this.textPaneSurrounding = new JPanel();
		this.textPaneSurrounding.setBorder(BorderFactory
				.createLineBorder(Color.DARK_GRAY));
		this.textPaneSurrounding.setBackground(new Color(255, 255, 220));
		this.textPaneSurrounding.setLayout(new BorderLayout());
		this.textPaneSurrounding.add(
				Box.createRigidArea(new Dimension(12, 12)), BorderLayout.NORTH);
		this.textPaneSurrounding.add(
				Box.createRigidArea(new Dimension(12, 12)), BorderLayout.SOUTH);
		this.textPaneSurrounding.add(
				Box.createRigidArea(new Dimension(12, 12)), BorderLayout.WEST);
		this.textPaneSurrounding.add(
				Box.createRigidArea(new Dimension(12, 12)), BorderLayout.EAST);

		this.scrollP = new JScrollPane(this.errorPanel);
		this.scrollP.setPreferredSize(new Dimension(700, 100));
		this.scrollP.getViewport().setBackground(Color.WHITE);
		this.textPaneSurrounding.add(this.scrollP, BorderLayout.CENTER);

		this.splitPane = new JSplitPane(JSplitPane.VERTICAL_SPLIT);
		this.splitPane.setBorder(BorderFactory
				.createLineBorder(Color.DARK_GRAY));
		this.splitPane.setTopComponent(this.editorPanelSurrounding);
		this.splitPane.setBottomComponent(this.textPaneSurrounding);
		this.splitPane.setOneTouchExpandable(true);
		this.splitPane.setContinuousLayout(true);
		this.splitPane.setResizeWeight(1);
		HierarchyListener hierarchyListener = new HierarchyListener() {
			public void hierarchyChanged(HierarchyEvent e) {
				long flags = e.getChangeFlags();
				if ((flags & HierarchyEvent.SHOWING_CHANGED) == HierarchyEvent.SHOWING_CHANGED) {
					Editor.this.splitPane.setDividerLocation(1.0);
				}
			}
		};
		this.splitPane.addHierarchyListener(hierarchyListener);

		this.mainPanel.add(this.splitPane, BorderLayout.CENTER);
		this.errorPaneAdded = true;
	}

	public void removeErrorPanel() {
		if (this.errorPaneAdded) {
			this.splitPane.setDividerLocation(1.0);
			this.errorPaneAdded = false;
		}
	}

	public void addErrorPanelAgain() {
		if (!this.errorPaneAdded) {
			this.splitPane.setDividerLocation(0.8);
			this.errorPaneAdded = true;
		}
	}

	public void addErrorPanel2() {
		JPanel textPaneSurrounding = new JPanel();
		textPaneSurrounding.setBorder(BorderFactory
				.createLineBorder(Color.DARK_GRAY));
		textPaneSurrounding.setBackground(new Color(255, 255, 220));
		textPaneSurrounding.setLayout(new BorderLayout());
		textPaneSurrounding.add(Box.createRigidArea(new Dimension(12, 12)),
				BorderLayout.NORTH);
		textPaneSurrounding.add(Box.createRigidArea(new Dimension(12, 12)),
				BorderLayout.SOUTH);
		textPaneSurrounding.add(Box.createRigidArea(new Dimension(12, 12)),
				BorderLayout.WEST);
		textPaneSurrounding.add(Box.createRigidArea(new Dimension(12, 12)),
				BorderLayout.EAST);

		this.errorPanel.setPreferredSize(new Dimension(700, 100));
		JScrollPane scrollP = new JScrollPane(this.errorPanel);
		textPaneSurrounding.add(scrollP, BorderLayout.CENTER);
		this.mainPanel.add(textPaneSurrounding, BorderLayout.CENTER);
	}

	protected void initMessagePanel() {
		this.add(this.messagePanel, BorderLayout.SOUTH);
	}

	public CompilerErrorPanel getErrorPanel() {
		return this.errorPanel;
	}

	public void changeIndent() {
		this.indent = !this.indent;
		this.indentMenuItem.setSelected(this.indent);
	}

	public boolean isIndent() {
		return this.indent;
	}

	public int getFontSize() {
		return this.fontSize;
	}

	public void setFontSize(int fontSize) {
		this.fontSize = fontSize;
		this.editorPanel.doc.changeFontSize(fontSize);
		this.lineNumbePanel.doc.changeFontSize(fontSize);
	}

	public JMenuItem getUndoMenuItem() {
		return undoMenuItem;
	}

	public JMenuItem getPrintMenuItem() {
		return printMenuItem;
	}

	public JMenuItem getCutMenuItem() {
		return cutMenuItem;
	}

	public JMenuItem getCopyMenuItem() {
		return copyMenuItem;
	}

	public JMenuItem getPasteMenuItem() {
		return pasteMenuItem;
	}

	public JCheckBoxMenuItem getIndentMenuItem() {
		return indentMenuItem;
	}

	public JMenu getFontsizeMenu() {
		return fontsizeMenu;
	}

	public JButton getPrintButton() {
		return printButton;
	}

	public JButton getCutButton() {
		return cutButton;
	}

	public JButton getCopyButton() {
		return copyButton;
	}

	public JButton getPasteButton() {
		return pasteButton;
	}

}

class MyCutAction extends DefaultEditorKit.CutAction {
	Editor editor;

	MyCutAction(Editor e) {
		this.editor = e;
	}

	public void actionPerformed(ActionEvent ev) {
		super.actionPerformed(ev);
		editor.getMessagePanel()
				.writeInfo(
						ResourceManager.getResourceManager().getValue(
								"msg.editor.cut"));
	}
}

class MyCopyAction extends DefaultEditorKit.CopyAction {
	Editor editor;

	MyCopyAction(Editor e) {
		this.editor = e;
	}

	public void actionPerformed(ActionEvent ev) {
		super.actionPerformed(ev);
		editor.getMessagePanel().writeInfo(
				ResourceManager.getResourceManager()
						.getValue("msg.editor.copy"));
	}
}

class MyPasteAction extends DefaultEditorKit.PasteAction {
	Editor editor;

	MyPasteAction(Editor e) {
		this.editor = e;
	}

	public void actionPerformed(ActionEvent ev) {
		super.actionPerformed(ev);
		editor.getMessagePanel().writeInfo(
				ResourceManager.getResourceManager().getValue(
						"msg.editor.paste"));
	}
}
