package compiler;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;
import java.io.FilenameFilter;
import java.io.IOException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Observable;
import java.util.Observer;

import javax.swing.JOptionPane;
import javax.tools.Diagnostic;
import javax.tools.DiagnosticCollector;
import javax.tools.JavaCompiler;
import javax.tools.JavaFileObject;
import javax.tools.StandardJavaFileManager;
import javax.tools.ToolProvider;

import listener.LoadStageListener;
import listener.SaveStageListener;
import model.ClassManager;
import model.Play;
import model.TheaterClass;
import simulation.SimulationManager;
import theater.Performance;
import theater_intern.IPerformance;
import util.PropertyManager;
import util.ResourceManager;
import editor.Editor;
import editor.EditorObservable;
import editor.SolistEditor;

/**
 * Verwaltet den Compiler und den ClassLoader
 * 
 * @author Dietrich Boles, Uni Oldenburg
 * @version 1.0 (12.11.2008)
 * 
 */
public class CompileManager extends Observable implements ActionListener,
		FilenameFilter, Observer {

	private final static String TMP_FILENAME = "SolistCompileTmpStage";

	private static CompileManager compileManager = null;

	private ClassLoader cl;
	private DiagnosticCollector<JavaFileObject> diagnostics;
	private boolean success;
	private boolean compilationRequired;
	private File tmpStageFile = null;

	private CompileManager() {
		this.cl = null;
		this.diagnostics = null;
		this.success = false;
		this.compilationRequired = false;
	}

	/**
	 * Liefert den CompileManager
	 * 
	 * @return
	 */
	public static CompileManager getCompileManager() {
		if (CompileManager.compileManager == null) {
			CompileManager.compileManager = new CompileManager();
		}
		return CompileManager.compileManager;
	}

	/**
	 * Liefert den aktuellen ClassLoader; gesetzt ist dieser erst nach dem
	 * ersten Compilieren
	 * 
	 * @return
	 */
	public ClassLoader getClassLoader() {
		return this.cl;
	}

	/**
	 * berprft, ob ein Compiler verfgbar ist
	 * 
	 * @return
	 */
	public boolean isCompilerAvailable() {
		return ToolProvider.getSystemJavaCompiler() != null;
	}

	public void actionPerformed(ActionEvent e) {
		IPerformance iPerf = Play.getPlay().getActivePerformance();
		Play play = Play.getPlay();

		if (play.isSimulator()) {
			try {
				iPerf.stopSimulation();
			} catch (Throwable th) {
			}
			SimulationManager.getSimulationManager().cancelSimulator();
			serializeTmp();
		} else {
			iPerf.resetSimulation();
		}

		// Flag: SolistDebugger
		// SolistDebugger.getSolistDebugger().writeEmptyInterface();
		//
		// if (Play.getPlay().getActiveStage() != null
		// && Play.getPlay().getActiveStage().getSolist() != null) {
		// this.handleSolistDebugInterface();
		// }

		boolean success = this.compile(Play.getPlay());

		this.handleCompilationResult(success);

		if (play.isSimulator()) {
			deserializeTmp();

			if (success) {
				Play.getPlay().getPlayFrame().getMessagePanel().writeInfo(
						ResourceManager.getResourceManager().getValue(
								"msg.compilationsim"));
			}
		} else {

			if (success) {
				Play.getPlay().getPlayFrame().getMessagePanel().writeInfo(
						ResourceManager.getResourceManager().getValue(
								"msg.compilation"));
			}
		}

		if (!success) {
			Play.getPlay().getPlayFrame().getMessagePanel().writeInfo(
					ResourceManager.getResourceManager().getValue(
							"msg.compilationerror"));
		}
	}

	/**
	 * Compiliert alle java-Dateien im angegebenen Verzeichnis. Liefert true,
	 * falls die Compilierung erfolgreich war. Falls die Compilierung nicht
	 * erfolgreich war, knnen Fehler ber die Methoden getFileErrorMsg und
	 * getOtherErrorMsg abgefragt werden.
	 * 
	 * @param dirName
	 * @return
	 */
	public boolean compile(Play play) {
		String dirName = play.getDirectory();
		File dir = new File(dirName);
		String[] fileNames = dir.list();

		// delete all class-files
		for (String fileName : fileNames) {
			if (fileName.endsWith(".class")) {
				File file = new File(dirName + File.separatorChar + fileName);
				file.delete();
			}
		}

		// save SolistProgramm
		try {
			SolistEditor.getSolistEditor().save();
		} catch (IOException exc) {
			if (!play.isSimulator()) {
				exc.printStackTrace();
			}
		}

		// compile all java-files
		this.diagnostics = new DiagnosticCollector<JavaFileObject>();
		JavaCompiler javac = ToolProvider.getSystemJavaCompiler();
		StandardJavaFileManager manager = javac.getStandardFileManager(
				this.diagnostics, null, null);

		List<File> javaFiles = this.getSourceFiles(play);
		// if (!withMain) {
		// ArrayList<File> cp = new ArrayList<File>(javaFiles.size());
		// for (File f : javaFiles) {
		// if (!f.getName().equals(SolistEditor.CLASS_NAME + ".java")) {
		// cp.add(f);
		// }
		// }
		// javaFiles = cp;
		// }

		this.success = true;
		if (javaFiles != null && javaFiles.size() > 0) {
			Iterable<? extends JavaFileObject> units = manager
					.getJavaFileObjectsFromFiles(javaFiles);
			this.success = javac.getTask(null, manager, this.diagnostics, null,
					null, units).call();
		}
		if (this.cl == null || this.success) {
			try {
				File dirFile = new File(dirName);
				File playPath = new File(dirFile.getAbsolutePath()
						+ File.separatorChar);
				URL[] urls = new URL[] { playPath.toURI().toURL() };
				this.cl = new URLClassLoader(urls);
			} catch (Throwable exc) {
				exc.printStackTrace();
			}
		}
		this.setCompilationRequired(!this.success);
		return this.success;
	}

	/**
	 * Liefert das Ergebnis der letzten Compilation
	 * 
	 * @return
	 */
	public boolean wasSuccessfully() {
		return this.success;
	}

	/**
	 * Liefert die Fehlermeldungen des Compilers zurck in Form eine Liste mit
	 * entsprechenden Fehlermeldungsobjekten. Kann dann aufgerufen werden, wenn
	 * die Methode compile den Wert false geliefert hat.
	 * 
	 * @param diagnostics
	 * @return
	 */
	public Collection<ArrayList<ErrorMessage>> getFileErrorMsg() {
		HashMap<String, ArrayList<ErrorMessage>> map = new HashMap<String, ArrayList<ErrorMessage>>();
		for (Diagnostic<? extends JavaFileObject> diagnostic : this.diagnostics
				.getDiagnostics()) {
			if (diagnostic.getKind() == Diagnostic.Kind.ERROR) {
				ErrorMessage msg = this.getErrorMsg(diagnostic);
				ArrayList<ErrorMessage> m = map.get(msg.getFilename());
				if (m == null) {
					ArrayList<ErrorMessage> l = new ArrayList<ErrorMessage>();
					l.add(msg);
					map.put(msg.getFilename(), l);
				} else {
					m.add(msg);
				}
			}
		}
		return map.values();
	}

	/**
	 * Falls es Fehlermeldungen gibt, die keiner existierenden Klasse zugeordnet
	 * werden knnen, wird ein entsprechender String mit der Meldung aufgebaut
	 * und zurck geliefert; ansonsten wird null geliefert. Kann dann aufgerufen
	 * werden, wenn die Methode compile den Wert false geliefert hat.
	 * 
	 * @param diagnostics
	 * @return
	 */
	public String getOtherErrorMsg() {
		String otherMsg = "";
		for (Diagnostic<? extends JavaFileObject> diagnostic : this.diagnostics
				.getDiagnostics()) {
			if (diagnostic.getKind() != Diagnostic.Kind.ERROR) {
				otherMsg += diagnostic.toString() + "\n";
			}
		}
		return otherMsg.equals("") ? null : otherMsg;
	}

	/**
	 * Markiert, ob neue Compilation erforderlich ist
	 * 
	 * @param required
	 */
	public void setCompilationRequired(boolean required) {
		// if (required != this.compilationRequired) {
		this.compilationRequired = required;
		this.setChanged();
		this.notifyObservers();
		this.clearChanged();
		// }
	}

	public boolean isCompilationRequired() {
		return this.compilationRequired;
	}

	/**
	 * Konvertiert die Diagnostic in eine ErrorMessage
	 * 
	 * @param diagnostic
	 * @return
	 */
	private ErrorMessage getErrorMsg(
			Diagnostic<? extends JavaFileObject> diagnostic) {
		ErrorMessage msg = new ErrorMessage();

		msg.setFilename(diagnostic.getSource().getName());
		msg.setMsg(diagnostic.getMessage(PropertyManager.getPropertyManager()
				.getLocale()));
		msg.setLine(diagnostic.getLineNumber());
		return msg;
	}

	public boolean accept(File path, String name) {
		return name.endsWith(".java");
	}

	public List<File> getSourceFiles(Play play) {
		File path = new File(play.getDirectory() + File.separatorChar)
				.getAbsoluteFile();
		return Arrays.asList(path.listFiles(this));
	}

	private void handleCompilationResult(boolean success) {
		Collection<TheaterClass> coll = ClassManager.getClassManager()
				.getAllClasses();
		for (TheaterClass cls : coll) {
			cls.getEditor().getErrorPanel().clear();
			cls.getEditor().removeErrorPanel();
		}
		SolistEditor.getSolistEditor().getErrorPanel().clear();
		Play.getPlay().getPlayFrame().getSolistEditorSplitPane()
				.setDividerLocation(1.0);

		if (!success) {
			String fileList = "";
			int x = 20;
			int y = 20;
			if (Play.getPlay().getPlayFrame() != null) {
				x = Play.getPlay().getPlayFrame().getX();
				y = Play.getPlay().getPlayFrame().getY();
			}
			Collection<ArrayList<ErrorMessage>> errors = this.getFileErrorMsg();
			for (ArrayList<ErrorMessage> errorList : errors) {
				fileList += errorList.get(0).getFilename() + "\n";
				String fileName = errorList.get(0).getFilename();
				fileName = fileName.substring(0, fileName.length() - 5);
				TheaterClass cls = ClassManager.getClassManager().getClass(
						fileName);
				Editor editor = null;
				if (cls != null) {
					editor = cls.getEditor();
					editor.getErrorPanel().setErrorMessages(errorList);
					if (!editor.isOpen()) {
						editor.setLocation(Math.max(20, x - editor.getWidth()
								/ 2), Math.max(20, y - editor.getHeight() / 2));
						x += 20;
						y += 20;
						if (!Play.getPlay().isSimulator()) {
							editor.open();
						}
					}
					editor.addErrorPanelAgain();
				} else {
					editor = SolistEditor.getSolistEditor().getEditor();
					editor.getErrorPanel().setErrorMessages(errorList);
					Play.getPlay().getPlayFrame().getSolistEditorSplitPane()
							.setDividerLocation(0.8);
				}
				editor.getErrorPanel().setErrorMessages(errorList);
			}

			if (errors.size() == 0) {
				String otherMsg = this.getOtherErrorMsg();
				if (otherMsg != null) {
					JOptionPane.showMessageDialog(
							Play.getPlay().getPlayFrame(), otherMsg,
							ResourceManager.getResourceManager().getValue(
									"compiler.title"),
							JOptionPane.ERROR_MESSAGE);
				}
			} else {
				JOptionPane.showMessageDialog(Play.getPlay().getPlayFrame(),
						ResourceManager.getResourceManager().getValue(
								"compiler.error")
								+ ": \n" + fileList, ResourceManager
								.getResourceManager()
								.getValue("compiler.title"),
						JOptionPane.ERROR_MESSAGE);
			}
			this.setCompilationRequired(true);
		} else {
			JOptionPane.showMessageDialog(Play.getPlay().getPlayFrame(),
					ResourceManager.getResourceManager().getValue(
							"compiler.success"), ResourceManager
							.getResourceManager().getValue("compiler.title"),
					JOptionPane.INFORMATION_MESSAGE);

			Play.getPlay().reset();

			this.setCompilationRequired(false);
		}
	}

	public void update(Observable o, Object arg) {
		EditorObservable edOb = (EditorObservable) o;
		if (edOb.getEditor().isSaved()) {
			this.setCompilationRequired(true);
		}
	}

	private void serializeTmp() {
		try {
			if (Performance.getPerformance().getActiveStage() != null) {
				this.tmpStageFile = File.createTempFile(TMP_FILENAME, "ser");
				SaveStageListener.saveStage(this.tmpStageFile);
			} else {
				this.tmpStageFile = null;
			}
		} catch (Throwable exc) {
			this.tmpStageFile = null;
			exc.printStackTrace();
		}
	}

	private void deserializeTmp() {
		if (this.tmpStageFile != null) {
			LoadStageListener.loadStage(this.tmpStageFile);
		}
	}

	// SolistDebugger
	// protected void handleSolistDebugInterface() {
	// boolean success = this.compile(Play.getPlay());
	// if (!success) {
	// return;
	// }
	// SolistDebugger.getSolistDebugger().writeFullInterface();
	// }
}
