import theater.*;


/**
 * Die Klasse stellt eine Repraesentation des virtuellen Spielers dar
 *
 * @author Dietrich Boles (Universitaet Oldenburg)
 * @version 1.0 (01.10.2009)
 *
 */
public class Koordinator extends Actor {
    // Default-Konstruktor (muss implementiert werden!)
    public Koordinator() {
        super(); // nicht loeschen!

        // der Spieler ist unsichtbar
        setVisible(false);
    }

    // Copy-Konstruktor (muss implementiert werden!)
    public Koordinator(Koordinator actor) {
        super(actor); // nicht loeschen!

        // der Spieler ist unsichtbar
        setVisible(false);
    }

    @Description("<html>In eine Reihe (erster Parameter, 1-9) und eine Spalte (zweiter Parameter, 1-9) " +
    "eine Zahl (dritter Parameter, 1-9) setzen." +
    "<br>Fehler, wenn der Spielzug nicht den Regeln entspricht.</html>")
    public void setzen(int reihe, int spalte, int zahl)
        throws UngueltigesFeldException, UngueltigeZahlException,
            FeldBelegtException, UngueltigerZugException {
        Performance.getPerformance().lock();

        try {
            checkFeld(reihe, spalte);
            checkZahl(zahl);
            checkBelegung(reihe, spalte);
            checkZug(reihe, spalte, zahl);

            Feld feld = ((Spielfeld) getStage()).getFeld();
            // feld.setZahl(reihe, spalte, zahl); erfolgt in add
            getStage().add(new Zahl(zahl), spalte - 1, reihe - 1);
        } finally {
            Performance.getPerformance().unlock();
        }
    }

    @Description("<html>Das Feld in einer Reihe (erster Parameter, 1-9) und einer Spalte (zweiter Parameter, 1-9) lschen." +
    "<br>Fehler, wenn das angegebene Feld nicht existiert.</html>")
    public void loeschen(int reihe, int spalte) throws UngueltigesFeldException {
        Performance.getPerformance().lock();

        try {
            checkFeld(reihe, spalte);

            Feld feld = ((Spielfeld) getStage()).getFeld();
            feld.loeschen(reihe, spalte);

            getStage()
                .remove(getStage()
                            .getComponentsAt(spalte - 1, reihe - 1, Zahl.class));
        } finally {
            Performance.getPerformance().unlock();
        }
    }

    @Description("<html>berprfen, ob ein Feld leer ist (erster Parameter = Reihe (1-9), zweiter Parameter = Spalte (1-9))." +
    "<br>Fehler, wenn das angegebene Feld nicht existiert.</html>")
    public boolean istLeer(int reihe, int spalte)
        throws UngueltigesFeldException {
        Performance.getPerformance().lock();

        try {
            checkFeld(reihe, spalte);

            Feld feld = ((Spielfeld) getStage()).getFeld();

            return feld.istLeer(reihe, spalte);
        } finally {
            Performance.getPerformance().unlock();
        }
    }

    @Description("<html>berprfen, ob ein Feld belegt ist (erster Parameter = Reihe (1-9), zweiter Parameter = Spalte (1-9))." +
    "<br>Fehler, wenn das angegebene Feld nicht existiert.</html>")
    public boolean istBelegt(int reihe, int spalte)
        throws UngueltigesFeldException {
        Performance.getPerformance().lock();

        try {
            checkFeld(reihe, spalte);

            Feld feld = ((Spielfeld) getStage()).getFeld();

            return feld.istBelegt(reihe, spalte);
        } finally {
            Performance.getPerformance().unlock();
        }
    }

    @Description("<html>berprfen, welche Zahl in einem bestimmten Feld platziert ist " +
    "(erster Parameter = Reihe (1-9), zweiter Parameter = Spalte (1-9))" +
    "<br>Fehler, wenn das angegebene Feld nicht existiert oder nicht belegt ist.</html>")
    public int welcheZahl(int reihe, int spalte)
        throws UngueltigesFeldException, FeldLeerException {
        Performance.getPerformance().lock();

        try {
            checkFeld(reihe, spalte);
            checkNichtBelegung(reihe, spalte);

            Feld feld = ((Spielfeld) getStage()).getFeld();

            return feld.getZahl(reihe, spalte);
        } finally {
            Performance.getPerformance().unlock();
        }
    }

    @Description("<html>Zeigt eine Lsung des Sudoku an.</html>")
    public void loesen() throws SudokuException {
        Performance.getPerformance().lock();
        Performance.getPerformance().freeze();

        try {
            Feld feld = ((Spielfeld) getStage()).getFeld();
            Feld solution = feld.loesen();

            if (solution == null) {
                throw new SudokuException("es gibt keine Lsung");
            } else {
                for (int reihe = 1; reihe <= 9; reihe++) {
                    for (int spalte = 1; spalte <= 9; spalte++) {
                        int zahl = solution.getZahl(reihe, spalte);

                        if (getStage()
                                    .getComponentsAt(spalte - 1, reihe - 1,
                                    Zahl.class).size() == 0) {
                            getStage().add(new Zahl(zahl), spalte - 1, reihe -
                                1);
                        }
                    }
                }
            }
        } finally {
            Performance.getPerformance().unfreeze();
            Performance.getPerformance().unlock();
        }
    }

    @Invisible
    private void checkFeld(int reihe, int spalte)
        throws UngueltigesFeldException {
        if ((reihe < 1) || (reihe > 9) || (spalte < 1) || (spalte > 9)) {
            throw new UngueltigesFeldException(reihe, spalte);
        }
    }

    @Invisible
    private void checkZahl(int zahl) throws UngueltigeZahlException {
        if ((zahl < 1) || (zahl > 9)) {
            throw new UngueltigeZahlException(zahl);
        }
    }

    @Invisible
    private void checkBelegung(int reihe, int spalte)
        throws FeldBelegtException {
        Feld feld = ((Spielfeld) getStage()).getFeld();

        if (feld.getZahl(reihe, spalte) != Feld.LEER) {
            throw new FeldBelegtException(reihe, spalte);
        }
    }

    @Invisible
    private void checkNichtBelegung(int reihe, int spalte)
        throws FeldLeerException {
        Feld feld = ((Spielfeld) getStage()).getFeld();

        if (feld.getZahl(reihe, spalte) == Feld.LEER) {
            throw new FeldLeerException(reihe, spalte);
        }
    }

    @Invisible
    private void checkZug(int reihe, int spalte, int zahl)
        throws UngueltigerZugException {
        Feld feld = ((Spielfeld) getStage()).getFeld();

        if (!feld.istZugMoeglich(reihe, spalte, zahl)) {
            throw new UngueltigerZugException(reihe, spalte, zahl);
        }
    }

    @Invisible
    public void print() {
        Feld feld = ((Spielfeld) getStage()).getFeld();
        System.out.println(feld.toString());
    }
}
