import theater.*;

import java.awt.Color;


/**
 * Die Klasse stellt eine Repraesentation der Schildkrte dar.
 * Hinweis: Groe Teile des Sourcecodes stammen aus dem Turtle-Szenario
 * von Greenfoot (www.greenfoot.org).
 *
 * @author Dietrich Boles (Universitaet Oldenburg)
 * @version 1.0 (30.09.2009)
 *
 */
public class Turtle extends Actor {
    /** The maximum value that the direction can have. */
    private final static double MAX_ANGLE = 360;

    /** Wheter the turtle should be visible or not. */
    private boolean visible = true;

    /** Whether the turtle should paint or not. */
    private boolean penDown = false;

    /** Colour of the pen. */
    private String color = "black";

    /** The direction the turtle is facing. */
    private double direction = 0.0;

    /** The x location. */
    private double x;

    /** The y location. */
    private double y;

    /** Image used when the pen is up */
    private TheaterImage penUpImage;

    /** Image used when the pen is down */
    private TheaterImage penDownImage;

    /** Image used when the turtle is not visible */
    private TheaterImage notVisibleImage;

    /** name of the turtle */
    protected Solist oscar;

    public Turtle() {
        super();
        setImage("turtle.png");
        penUpImage = getImage();
        penDownImage = new TheaterImage(penUpImage);
        drawPen(penDownImage);
        pendown();
        oscar = (Solist) this;
    }

    public Turtle(Turtle turtle) {
        super(turtle);
        this.penDown = turtle.penDown;
        this.color = turtle.color;
        this.direction = turtle.direction;
        this.x = turtle.x;
        this.y = turtle.y;
        this.penUpImage = turtle.penUpImage;
        this.penDownImage = turtle.penDownImage;
        this.notVisibleImage = turtle.notVisibleImage;
        oscar = (Solist) this;
    }

    @Description("Die Schildkrte bewegt sich zur Mitte des Bildschirms mit Ausrichtung nach oben (0 Grad)")
    public void home() {
        Performance.getPerformance().lock();

        try {
            moveTo(getStage().getNumberOfColumns() / 2,
                getStage().getNumberOfRows() / 2);
            setRotation(0);
        } finally {
            Performance.getPerformance().unlock();
        }
    }

    @Description("Der Bildschirm wird gelscht, die Position der Schildkrte ndert sich nicht")
    public void clean() {
        Performance.getPerformance().lock();

        try {
            ((PaintArea) getStage()).clean();
        } finally {
            Performance.getPerformance().unlock();
        }
    }

    @Description("home und clean zusammen")
    public void cleanscreen() {
        Performance.getPerformance().lock();
        Performance.getPerformance().freeze();

        try {
            home();
            clean();
        } finally {
            Performance.getPerformance().unfreeze();
            Performance.getPerformance().unlock();
        }
    }

    @Description("Die Schildkrte wird unsichtbar")
    public void hideturtle() {
        Performance.getPerformance().lock();

        try {
            visible = false;
            setImage(notVisibleImage);
        } finally {
            Performance.getPerformance().unlock();
        }
    }

    @Description("Die Schildkrte wird sichtbar")
    public void showturtle() {
        Performance.getPerformance().lock();

        try {
            visible = true;

            if (penDown) {
                setImage(penDownImage);
            } else {
                setImage(penUpImage);
            }
        } finally {
            Performance.getPerformance().unlock();
        }
    }

    @Description("<html>Der Schildkrte wird eine Zeichenfarbe zugewiesen.<br>" +
    "Erlaubte Farben sind red, black, blue, yellow, green, magenta und white</html>")
    public void setpencolor(String newColor) {
        Performance.getPerformance().lock();

        try {
            color = newColor;
            drawPen(penDownImage);
        } finally {
            Performance.getPerformance().unlock();
        }
    }

    @Description("Der Stift wird von der Zeichenflche genommen")
    public void penup() {
        Performance.getPerformance().lock();

        try {
            penDown = false;

            if (visible) {
                setImage(penUpImage);
            }
        } finally {
            Performance.getPerformance().unlock();
        }
    }

    @Description("Der Stift wird auf die Zeichenflche gesetzt")
    public void pendown() {
        Performance.getPerformance().lock();

        try {
            penDown = true;

            if (visible) {
                setImage(penDownImage);
            }
        } finally {
            Performance.getPerformance().unlock();
        }
    }

    @Description("Die Schildkrte dreht sich um einen bestimmten Winkel nach rechts")
    public void right(double degrees) {
        Performance.getPerformance().lock();

        try {
            if (degrees < 0) {
                left(-degrees);
            } else {
                direction = direction + degrees;

                if (direction > MAX_ANGLE) {
                    direction = direction % MAX_ANGLE;
                }

                setRotation((int) direction);
            }
        } finally {
            Performance.getPerformance().unlock();
        }
    }

    @Description("Die Schildkrte dreht sich um einen bestimmten Winkel nach links")
    public void left(double degrees) {
        Performance.getPerformance().lock();

        try {
            if (degrees < 0) {
                right(-degrees);
            } else {
                degrees = degrees % MAX_ANGLE;
                direction = direction - degrees;

                if (direction < 0) {
                    direction = MAX_ANGLE + direction;
                }

                setRotation((int) direction);
            }
        } finally {
            Performance.getPerformance().unlock();
        }
    }

    @Description("Die Schildkrte bewegt sich um eine bestimmte Anzahl von Einheiten nach vorne")
    public void forward(double distance) {
        Performance.getPerformance().lock();

        try {
            if (distance < 0) {
                back(-distance);
            } else {
                double directionRad = Math.toRadians(correctDirection());
                double xDist = distance * Math.cos(directionRad);
                double yDist = distance * Math.sin(directionRad);
                moveTo(x + xDist, y + yDist);
            }
        } finally {
            Performance.getPerformance().unlock();
        }
    }

    @Description("Die Schildkrte bewegt sich um eine bestimmte Anzahl von Einheiten nach hinten")
    public void back(double distance) {
        Performance.getPerformance().lock();

        try {
            if (distance < 0) {
                forward(-distance);
            } else {
                double directionRad = Math.toRadians(correctDirection());
                double xDist = distance * Math.cos(directionRad);
                double yDist = distance * Math.sin(directionRad);
                moveTo(x - xDist, y - yDist);
            }
        } finally {
            Performance.getPerformance().unlock();
        }
    }

    /* * * * * * * * * * * * * * * * * * * * * * * * * * * * */

    // We need to make sure that our own representaion of the location is the
    // same as the Stage's.
    @Invisible
    public final void addedToStage(Stage stage) {
        x = getColumn();
        y = getRow();
        notVisibleImage = new TheaterImage(penUpImage);
        changeNotVisibleColor();
    }

    // Set the location of the turtle.
    @Invisible
    public final void setLocation(double x, double y) {
        this.x = x;
        this.y = y;
        super.setLocation((int) Math.floor(x), (int) Math.floor(y));
    }

    // We need to override this method, so we can interactively move objects.
    // This method should not be used by subclasses. Use the 
    // setLocation(double x, double y) instead
    @Invisible
    public final void setLocation(int x, int y) {
        this.x = x;
        this.y = y;
        super.setLocation(x, y);
    }

    // Moves the turtle to the given position.
    @Invisible
    void moveTo(double newX, double newY) {
        double oldX = x;
        double oldY = y;

        if (penDown) {
            Performance.getPerformance().freeze();
            setLocation(newX, newY);
            drawLine(oldX, oldY, newX, newY);
            Performance.getPerformance().unfreeze();
        } else {
            setLocation(newX, newY);
        }
    }

    // Draw a line between two points with the current colour.
    @Invisible
    void drawLine(double x1, double y1, double x2, double y2) {
        TheaterImage image = getStage().getBackground();
        Color awtColor = decode(color);
        image.setColor(awtColor);
        image.drawLine((int) Math.ceil(x1), (int) Math.ceil(y1),
            (int) Math.ceil(x2), (int) Math.ceil(y2));
    }

    // Draw the pen on the back of the turtle with the correct colour 
    @Invisible
    void drawPen(TheaterImage image) {
        double halfWidth = image.getWidth() / 2.;
        double halfHeight = image.getHeight() / 2.;
        int penWidth = (int) halfWidth / 2;
        int penHeight = (int) halfHeight / 2;
        int penX = (int) (halfWidth - (penWidth / 2));
        int penY = (int) (halfHeight - (penHeight / 2));
        Performance.getPerformance().freeze();

        Color awtColor = decode(color);
        image.setColor(awtColor);
        image.fillOval(penX - 1, penY + 1, penWidth, penHeight);
        Performance.getPerformance().unfreeze();
    }

    // Translate a String into a Color
    @Invisible
    Color decode(String colorString) {
        if (colorString.equals("red")) {
            return Color.red;
        } else if (colorString.equals("black")) {
            return Color.black;
        } else if (colorString.equals("blue")) {
            return Color.blue;
        } else if (colorString.equals("yellow")) {
            return Color.yellow;
        } else if (colorString.equals("green")) {
            return Color.green;
        } else if (colorString.equals("magenta")) {
            return Color.magenta;
        } else if (colorString.equals("white")) {
            return Color.white;
        } else {
            return Color.black;
        }
    }

    @Invisible
    double correctDirection() {
        double dir = direction - 90.0;

        if (dir < 0) {
            dir = MAX_ANGLE + dir;
        }

        return dir;
    }

    @Invisible
    void changeNotVisibleColor() {
        notVisibleImage.setColor(((PaintArea) getStage()).getBackgroundColor());
        notVisibleImage.fill();
    }
}
