package view;

import java.awt.AlphaComposite;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.EventQueue;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.Point;
import java.awt.Rectangle;
import java.awt.geom.AffineTransform;
import java.lang.reflect.InvocationTargetException;
import java.util.List;

import javax.swing.JFrame;
import javax.swing.JPanel;
import javax.swing.JScrollPane;

import listener.StoppedMouseListener;
import listener.TheaterMouseListener;
import model.Play;
import simulation.listener.TheaterKeyListener;
import theater.TheaterImage;
import theater_intern.IComponent;
import theater_intern.IStage;

public class StagePanel extends JPanel {

	final static int BORDER = 40;

	private Play play;
	private Image stageImage;

	private TheaterMouseListener mouseListener;
	private TheaterKeyListener keyListener;

	private Painter painter;

	/**
	 * Erzeugt ein StagePanel
	 * 
	 * @param p
	 */
	public StagePanel(Play p) {
		super();
		this.play = p;
		this.stageImage = null;
		this.setPreferredSize(this.calcDimension());
		this.setFocusable(true);
		this.setBackground(new Color(252, 255, 206));
		painter = new Painter(this);
		this.mouseListener = new StoppedMouseListener();
		this.addMouseListener(this.mouseListener);
		this.addMouseMotionListener(this.mouseListener);
		this.keyListener = null;
	}

	/**
	 * Zeichnet den StagePanel
	 */

	public void repaintStage() {
		if (play.getActivePerformance().frozen()) {
			return;
		}
		try {
			if (EventQueue.isDispatchThread()) {
				// EventQueue.invokeLater(painter);
				this.repaintStageIntern();
			} else {
				if (play.isEventQueueActive()) {
					EventQueue.invokeLater(painter);
				} else {
					/*
					 * prinzipieller Fehler, wenn zwischer der active-Abfrage
					 * und diesem Aufruf eine Benutzeraktion erfolgt, deren
					 * Listener auf den aufrufenden Thread wartet -> Deadlock
					 * Eine Synchonisation ist aber nicht mglich, da diese
					 * prinzipiell in dem EventQueue-Thread erfolgen msste
					 */
					EventQueue.invokeAndWait(painter);
				}
			}

		} catch (InterruptedException exc) {
			Thread.currentThread().interrupt();
		} catch (InvocationTargetException exc) {
		}
	}

	public void repaintStage2() {
		this.repaintStageIntern();
	}

	void repaintStageIntern() {
		Dimension newD = this.calcDimension();
		Dimension oldD = this.getPreferredSize();
		boolean newDim;
		if (newDim = !oldD.equals(newD)) {
			JScrollPane scrollPane = this.play.getPlayFrame()
					.getStageScrollPane();
			this.setPreferredSize(newD);
			scrollPane.remove(this);
			scrollPane.setViewportView(this);
			JFrame frame = this.play.getPlayFrame();
			frame.pack();
		}
		this.createStageImage(newDim);
		this.repaint(calcStageRectangle());
	}

	private void createStageImage(boolean newDim) {
		IStage stage = this.play.getActiveStage();
		if (stage == null) {
			this.stageImage = null;
			return;
		}

		int cellSize = stage.getCellSize();
		int w = stage.getNumberOfColumns() * cellSize;
		int h = stage.getNumberOfRows() * cellSize;
		int x = 1;
		int y = 1;

		if (this.stageImage == null || newDim) {
			this.stageImage = this.createImage(w + 2, h + 2);
		}
		Graphics2D g = (Graphics2D) this.stageImage.getGraphics();

		// g.clearRect(0, 0, w + 2, h + 2);

		g.setColor(Color.WHITE);
		g.fillRect(x, y, w, h);
		g.setColor(Color.BLACK);
		g.drawRect(x - 1, y - 1, w + 1, h + 1);
		g.clipRect(x, y, w, h);

		// draw background
		TheaterImage bgIm = stage.getBackground();
		if (bgIm != null) {
			g.setComposite(AlphaComposite.getInstance(AlphaComposite.SRC_OVER,
					bgIm.getTransparency() / 255.0f));
			Image backgroundImage = bgIm.getAwtImage();
			if (backgroundImage != null) {
				if (!stage.isTiled()) {
					g.drawImage(backgroundImage, x, y, this);
				} else {
					int bw = backgroundImage.getWidth(this);
					int bh = backgroundImage.getHeight(this);
					for (int c = 0; x + c * bw < x + w; c++) {
						for (int r = 0; y + r * bh < y + h; r++) {
							g.drawImage(backgroundImage, x + c * bw,
									y + r * bh, this);
						}
					}

				}
			}
		}

		// draw components
		List<IComponent> comps = stage.getComponentsInPaintOrder();
		for (IComponent comp : comps) {
			if (comp.isVisible()) {
				int c = comp.getColumn();
				int r = comp.getRow();
				TheaterImage tImg = comp.getImage();
				Image img = tImg.getAwtImage();
				int ox = x + c * cellSize;
				ox = ox - (img.getWidth(this) / 2 - cellSize / 2);
				int oy = y + r * cellSize;
				oy = oy - (img.getHeight(this) / 2 - cellSize / 2);
				g.setComposite(AlphaComposite.getInstance(
						AlphaComposite.SRC_OVER,
						tImg.getTransparency() / 255.0f));
				if (comp.getRotation() == 0) {
					g.drawImage(img, ox, oy, this);
				} else {

					AffineTransform origTrans = g.getTransform();
					AffineTransform newTrans = (AffineTransform) origTrans
							.clone();

					int xRot = x + c * cellSize + cellSize / 2;
					int yRot = y + r * cellSize + cellSize / 2;
					newTrans.rotate(Math.toRadians(comp.getRotation()), xRot,
							yRot);
					g.setTransform(newTrans);

					g.drawImage(img, ox, oy, this);
					g.setTransform(origTrans);
				}
			}
		}
	}

	synchronized protected void paintComponent(Graphics g) {
		super.paintComponent(g);
		if (this.play.getActiveStage() == null || this.stageImage == null) {
			return;
		}
		int cellSize = this.play.getActiveStage().getCellSize();
		int w = this.play.getActiveStage().getNumberOfColumns() * cellSize;
		int h = this.play.getActiveStage().getNumberOfRows() * cellSize;
		int x = this.getWidth() / 2 - w / 2;
		int y = this.getHeight() / 2 - h / 2;
		g.drawImage(this.stageImage, x, y, this);
	}

	/**
	 * Lscht das StagePanel
	 */
	public void destroyImage() {
		this.stageImage = null;
		this.repaintStage();
	}

	/**
	 * Setzt den Focus auf das StagePanel
	 */
	public void setFocus() {
		this.requestFocusInWindow();
	}

	/**
	 * Liefert die Reihe der Zelle, in der sich das Pixel (in
	 * Swing-Panel-Koordinaten) befindet; (auch auerhalb der Bhne)
	 * 
	 * @param x
	 * @return
	 */
	public int getCellRow(int y) {
		int fak = 1;
		int cellSize = this.play.getActiveStage().getCellSize();
		int ph = this.play.getActiveStage().getNumberOfRows() * cellSize;
		int py = (this.getHeight() - ph) / 2; // Beginn der eigentlichen Stage
		y = y - py;
		if (y < 0) {
			fak = -1;
			y = -y + cellSize;
		}
		return fak * (y / cellSize);
	}

	/**
	 * Liefert die Spalte der Zelle, in der sich das Pixel (in
	 * Swing-Panel-Koordinaten) befindet (auch auerhalb der Bhne)
	 * 
	 * @param x
	 * @return
	 */
	public int getCellColumn(int x) {
		int fak = 1;
		int cellSize = this.play.getActiveStage().getCellSize();
		int pw = this.play.getActiveStage().getNumberOfColumns() * cellSize;
		int px = (this.getWidth() - pw) / 2; // Beginn der eigentlichen Stage
		x = x - px;
		if (x < 0) {
			fak = -1;
			x = -x + cellSize;
		}
		return fak * (x / cellSize);
	}

	/**
	 * Liefert den Bereich, den das Bild der Componente berdeckt (in
	 * Swing-Panel-Koordinaten)
	 * 
	 * @param comp
	 * @return
	 */
	public Rectangle getRectangle(IComponent comp) {
		int cellSize = this.play.getActiveStage().getCellSize();
		int w = this.play.getActiveStage().getNumberOfColumns() * cellSize;
		int h = this.play.getActiveStage().getNumberOfRows() * cellSize;
		int x = (this.getWidth() - w) / 2; // Beginn der eigentlichen Stage
		int y = (this.getHeight() - h) / 2; // Beginn der eigentlichen Stage
		int c = comp.getColumn();
		int r = comp.getRow();
		Image img = comp.getImage().getAwtImage();
		int ox = x + c * cellSize; // Beginn der Zelle
		ox = ox - (img.getWidth(this) - cellSize) / 2;
		// minus halbe Breite des Bildes (Anmerkung: Das Bild ist immer in der
		// Mitte einer Zelle platziert)
		int oy = y + r * cellSize;
		oy = oy - (img.getHeight(this) - cellSize) / 2;
		return new Rectangle(ox, oy, img.getWidth(this), img.getHeight(this));
	}

	/**
	 * Liefert den Bereich, den die Bhne berdeckt (in Swing-Panel-Koordinaten)
	 * 
	 * @return
	 */
	public Rectangle getStageRectangle() {
		int cellSize = this.play.getActiveStage().getCellSize();
		int w = this.play.getActiveStage().getNumberOfColumns() * cellSize;
		int h = this.play.getActiveStage().getNumberOfRows() * cellSize;
		int x = (this.getWidth() - w) / 2; // Beginn der eigentlichen Stage
		int y = (this.getHeight() - h) / 2; // Beginn der eigentlichen Stage
		return new Rectangle(x, y, w, h);
	}

	/**
	 * Liefert den Bereich, den eine Zelle berdeckt (in Stage-Koordinaten)
	 * 
	 * @param row
	 * @param col
	 * @return
	 */
	public Rectangle getCellRectangle(int row, int col) {
		int cellSize = this.play.getActiveStage().getCellSize();
		int ox = col * cellSize;
		int oy = row * cellSize;
		return new Rectangle(ox, oy, cellSize, cellSize);
	}

	/**
	 * Liefert den Bereich, den das Bild der Componente berdeckt (in
	 * Stage-Koordinaten)
	 * 
	 * @param comp
	 * @return
	 */
	public Rectangle getStageRectangle(IComponent comp) {
		int cellSize = this.play.getActiveStage().getCellSize();
		int c = comp.getColumn();
		int r = comp.getRow();
		Image img = comp.getImage().getAwtImage();
		int ox = c * cellSize;
		ox = ox - (img.getWidth(this) - cellSize) / 2;
		int oy = r * cellSize;
		oy = oy - (img.getHeight(this) - cellSize) / 2;
		return new Rectangle(ox, oy, img.getWidth(this), img.getHeight(this));
	}

	/**
	 * Liefert die Stage-X-Koordinate bez. der bergebenen Panel-X-Koordinate
	 * 
	 * @param panelX
	 * @return
	 */
	public int getStageX(int panelX) {
		IStage stage = this.play.getActiveStage();
		int cellSize = stage.getCellSize();
		int pw = stage.getNumberOfColumns() * cellSize;
		return panelX - (this.getWidth() - pw) / 2; // abzglich Rand
	}

	/**
	 * Liefert die Stage-Y-Koordinate bez. der bergebenen Panel-Y-Koordinate
	 * 
	 * @param panelY
	 * @return
	 */
	public int getStageY(int panelY) {
		IStage stage = this.play.getActiveStage();
		int cellSize = stage.getCellSize();
		int ph = stage.getNumberOfRows() * cellSize;
		return panelY - (this.getHeight() - ph) / 2; // abzglich Rand
	}

	public Point getDrawPoint(Point panelPoint, Image img) {
		int cellSize = this.play.getActiveStage().getCellSize();
		int w = this.play.getActiveStage().getNumberOfColumns() * cellSize;
		int h = this.play.getActiveStage().getNumberOfRows() * cellSize;
		int x = (this.getWidth() - w) / 2; // Beginn der eigentlichen Stage
		int y = (this.getHeight() - h) / 2; // Beginn der eigentlichen Stage
		int c = getCellColumn(panelPoint.x);
		int r = getCellRow(panelPoint.y);
		int ox = x + c * cellSize; // Beginn der Zelle
		ox = ox - (img.getWidth(this) - cellSize) / 2;
		// minus halbe Breite des Bildes (Anmerkung: Das Bild ist immer in der
		// Mitte einer Zelle platziert)
		int oy = y + r * cellSize;
		oy = oy - (img.getHeight(this) - cellSize) / 2;
		return new Point(ox, oy);
	}

	public boolean isInsideStage(Point panelPoint) {
		Rectangle rect = getStageRectangle();
		return rect.contains(panelPoint);
	}

	/**
	 * Berechnet die Gre des StagePanels
	 * 
	 * @return
	 */
	private Dimension calcDimension() {
		IStage stage = this.play.getActiveStage();
		if (stage != null) {
			int cellSize = this.play.getActiveStage().getCellSize();
			int w = stage.getNumberOfColumns() * cellSize;
			int h = stage.getNumberOfRows() * cellSize;
			return new Dimension(w + StagePanel.BORDER, h + StagePanel.BORDER);
		} else {
			return new Dimension(StagePanel.BORDER, StagePanel.BORDER);
		}
	}

	private Rectangle calcStageRectangle() {
		IStage stage = this.play.getActiveStage();
		if (stage != null) {
			int cellSize = this.play.getActiveStage().getCellSize();
			int w = getWidth();
			int h = getHeight();
			int sw = stage.getNumberOfColumns() * cellSize;
			int sh = stage.getNumberOfRows() * cellSize;
			return new Rectangle((w - sw) / 2 - 1, (h - sh) / 2 - 1, sw + 2,
					sh + 2);
		} else {
			return new Rectangle(0, 0, getWidth(), getHeight());
		}
	}

	public TheaterMouseListener getMouseListener() {
		return this.mouseListener;
	}

	public void setMouseListener(TheaterMouseListener listener) {
		this.removeMouseListener(this.mouseListener);
		this.removeMouseMotionListener(this.mouseListener);
		this.mouseListener = listener;
		this.addMouseListener(this.mouseListener);
		this.addMouseMotionListener(this.mouseListener);
	}

	public void setKeyListener(TheaterKeyListener listener) {
		if (this.keyListener != null) {
			this.removeKeyListener(this.keyListener);
		}
		this.keyListener = listener;
		if (this.keyListener != null) {
			this.addKeyListener(this.keyListener);
		}
	}

	public boolean imageUpdate(Image img, int flags, int x, int y, int w, int h) {
//		 System.out.println("Image update: flags="+flags+
//		 " x="+x+" y="+y+" w="+w+" h="+h);
		repaintStage();
		return true;
	}

	public Image getStageImage() {
		return stageImage;
	}

}

class Painter implements Runnable {

	private StagePanel panel;

	Painter(StagePanel p) {
		this.panel = p;
	}

	public void run() {
		this.panel.repaintStageIntern();
	}
}