package theater;

import model.Play;
import theater_intern.IComponent;
import theater_intern.IStage;
import util.Utils;

/**
 * Die Klasse Rectangle reprsentiert ein rechteckiges Gebiet auf der Bhne. Sie
 * implementiert das Interface PixelArea, so dass mit dieser Klasse Kollisionen
 * von rechteckigen Gebieten mit anderen Gebieten der Bhne berprft werden
 * knnen.
 * 
 * @author Dietrich Boles, Universitt Oldenburg, Germany
 * @version 1.0 (17.07.2009)
 * 
 */
public class Rectangle implements PixelArea {

	/**
	 * x-Koordinate der linken oberen Ecke
	 */
	protected int x;

	/**
	 * y-Koordinate der linken oberen Ecke
	 */
	protected int y;

	/**
	 * Breite des Rechteckes
	 */
	protected int width;

	/**
	 * Hhe des Rechteckes
	 */
	protected int height;

	/**
	 * Konstruktor zum Initialisieren eines Rechteckes
	 * 
	 * @param x
	 *            x-Koordinate der linken oberen Ecke
	 * @param y
	 *            y-Koordinate der linken oberen Ecke
	 * @param w
	 *            Breite des Rechteckes
	 * @param h
	 *            Hhe des Rechteckes
	 */
	public Rectangle(int x, int y, int w, int h) {
		Play.getPlay().getActivePerformance().checkStop();
		Performance.getPerformance().lock();
		try {
			this.x = x;
			this.y = y;
			this.width = w;
			this.height = h;
		} finally {
			Performance.getPerformance().unlock();
		}
	}

	/**
	 * Konstruktor zum Initialisieren eines Rechteckes mit einem
	 * java.awt.Rectangle-Objekt
	 * 
	 * @param r
	 *            ein bereits existierendes java.awt.Rectangle-Objekt (draf
	 *            nicht null sein)
	 */
	public Rectangle(java.awt.Rectangle r) {
		this(r.x, r.y, r.width, r.height);
	}

	/**
	 * berprft, ob der Punkt mit den als Parameter bergebenen Koordinaten
	 * innerhalb des aufgerufenen Rechteckes liegt.
	 * 
	 * @param x
	 *            x-Koordinate des Punktes
	 * @param y
	 *            y-Koordinate des Punktes
	 * @return genau dann true, wenn der Punkt mit den als Parameter bergebenen
	 *         Koordinaten innerhalb des aufgerufenen Rechteckes liegt
	 * 
	 * @see theater.PixelArea#contains(int, int)
	 */
	public boolean contains(int x, int y) {
		Play.getPlay().getActivePerformance().checkStop();
		Performance.getPerformance().lock();
		try {
			return x >= this.x && x <= this.x + this.width && y >= this.y
					&& y <= this.y + this.height;
		} finally {
			Performance.getPerformance().unlock();
		}
	}

	/**
	 * berprft, ob das aufgerufene Rechteck komplett innerhalb der als
	 * Parameter bergebenen PixelArea liegt.
	 * 
	 * @param area
	 *            die zu vergleichende PixelArea
	 * @return genau dann true, wenn das aufgerufene Rechteck komplett innerhalb
	 *         der als Parameter bergebenen PixelArea liegt
	 * 
	 * @see theater.PixelArea#isInside(theater.PixelArea)
	 */
	public boolean isInside(PixelArea area) {
		// aus Performancegruenden
		Play.getPlay().getActivePerformance().checkStop();
		Performance.getPerformance().lock();
		try {
			if (Utils.dynInstanceof(area, Component.class)) {
				java.awt.Rectangle compRect = IComponent.getIComp(
						(Component) area).getImageRectangle();
				return compRect.contains(this.toAWTRectangle());
			} else if (Utils.dynInstanceof(area, theater.Rectangle.class)) {
				java.awt.Rectangle pixelRect = ((theater.Rectangle) area)
						.toAWTRectangle();
				return pixelRect.contains(this.toAWTRectangle());
			} else if (Utils.dynInstanceof(area, theater.Point.class)) {
				java.awt.Point p = ((theater.Point) area).toAWTPoint();
				return this.x == p.x && this.y == p.y && this.width == 1
						&& this.height == 1;
			} else if (Utils.dynInstanceof(area, Stage.class)) {
				IStage stage = IStage.getIStage((Stage) area);
				java.awt.Rectangle compRect = new java.awt.Rectangle(0, 0,
						stage.getNumberOfColumns() * stage.getCellSize(), stage
								.getNumberOfRows()
								* stage.getCellSize());
				return compRect.contains(this.toAWTRectangle());
			} else if (Utils.dynInstanceof(area, Cell.class)) {
				Cell cell = (Cell) area;
				java.awt.Rectangle rect = Play.getPlay().getStagePanel()
						.getCellRectangle(cell.getRow(), cell.getCol());
				return rect.contains(this.toAWTRectangle());
			} else if (Utils.dynInstanceof(area, CellArea.class)) {
				CellArea cArea = (CellArea) area;
				java.awt.Rectangle rect = Play.getPlay().getStagePanel()
						.getCellRectangle(cArea.getFromRow(),
								cArea.getFromCol());
				java.awt.Rectangle rect2 = new java.awt.Rectangle(rect.x,
						rect.y, rect.width * cArea.getNumberOfCols(),
						rect.height * cArea.getNumberOfRows());
				return rect2.contains(this.toAWTRectangle());
			} else {
				for (int x = this.x; x < this.x + this.width; x++) {
					for (int y = this.y; y < this.y + this.height; y++) {
						if (!area.contains(x, y)) {
							return false;
						}
					}
				}
				return true;
			}
		} finally {
			Performance.getPerformance().unlock();
		}
	}

	/**
	 * berprft, ob das aufgerufene Rechteck die als Parameter bergebene
	 * PixelArea schneidet.
	 * 
	 * @param area
	 *            die zu vergleichende PixelArea
	 * @return genau dann true, wenn das aufgerufene Rechteck die als Parameter
	 *         bergebene PixelArea schneidet
	 * @see theater.PixelArea#intersects(theater.PixelArea)
	 */
	public boolean intersects(PixelArea area) {
		// aus Performancegruenden
		Play.getPlay().getActivePerformance().checkStop();
		Performance.getPerformance().lock();
		try {
			if (Utils.dynInstanceof(area, Component.class)) {
				java.awt.Rectangle compRect = IComponent.getIComp(
						(Component) area).getImageRectangle();
				return this.toAWTRectangle().intersects(compRect);
			} else if (Utils.dynInstanceof(area, theater.Rectangle.class)) {
				java.awt.Rectangle pixelRect = ((theater.Rectangle) area)
						.toAWTRectangle();
				return this.toAWTRectangle().intersects(pixelRect);
			} else if (Utils.dynInstanceof(area, theater.Point.class)) {
				java.awt.Point p = ((theater.Point) area).toAWTPoint();
				return this.toAWTRectangle().contains(p);
			} else if (Utils.dynInstanceof(area, Stage.class)) {
				IStage stage = IStage.getIStage((Stage) area);
				java.awt.Rectangle compRect = new java.awt.Rectangle(0, 0,
						stage.getNumberOfColumns() * stage.getCellSize(), stage
								.getNumberOfRows()
								* stage.getCellSize());
				return this.toAWTRectangle().intersects(compRect);
			} else if (Utils.dynInstanceof(area, Cell.class)) {
				Cell cell = (Cell) area;
				java.awt.Rectangle rect = Play.getPlay().getStagePanel()
						.getCellRectangle(cell.getRow(), cell.getCol());
				return this.toAWTRectangle().intersects(rect);
			} else if (Utils.dynInstanceof(area, CellArea.class)) {
				CellArea cArea = (CellArea) area;
				java.awt.Rectangle rect = Play.getPlay().getStagePanel()
						.getCellRectangle(cArea.getFromRow(),
								cArea.getFromCol());
				java.awt.Rectangle rect2 = new java.awt.Rectangle(rect.x,
						rect.y, rect.width * cArea.getNumberOfCols(),
						rect.height * cArea.getNumberOfRows());

				return this.toAWTRectangle().intersects(rect2);
			} else {
				for (int x = this.x; x < this.x + this.width; x++) {
					for (int y = this.y; y < this.y + this.height; y++) {
						if (area.contains(x, y)) {
							return true;
						}
					}
				}
				return false;
			}
		} finally {
			Performance.getPerformance().unlock();
		}
	}

	/**
	 * Liefert die x-Koordinate der linken oberen Ecke des Rechteckes.
	 * 
	 * @return die x-Koordinate der linken oberen Ecke des Rechteckes
	 */
	public int getX() {
		Play.getPlay().getActivePerformance().checkStop();
		Performance.getPerformance().lock();
		try {
			return this.x;
		} finally {
			Performance.getPerformance().unlock();
		}
	}

	/**
	 * Liefert die y-Koordinate der linken oberen Ecke des Rechteckes.
	 * 
	 * @return die y-Koordinate der linken oberen Ecke des Rechteckes
	 */
	public int getY() {
		Play.getPlay().getActivePerformance().checkStop();
		Performance.getPerformance().lock();
		try {
			return this.y;
		} finally {
			Performance.getPerformance().unlock();
		}
	}

	/**
	 * Liefert die Breite des Rechteckes.
	 * 
	 * @return die Breite des Rechteckes
	 */
	public int getWidth() {
		Play.getPlay().getActivePerformance().checkStop();
		Performance.getPerformance().lock();
		try {
			return this.width;
		} finally {
			Performance.getPerformance().unlock();
		}
	}

	/**
	 * Liefert die Hhe des Rechteckes.
	 * 
	 * @return die Hhe des Rechteckes
	 */
	public int getHeight() {
		Play.getPlay().getActivePerformance().checkStop();
		Performance.getPerformance().lock();
		try {
			return this.height;
		} finally {
			Performance.getPerformance().unlock();
		}
	}

	/**
	 * Wandelt das Rechteck um in ein Objekt der Klasse java.awt.Rectangle
	 * 
	 * @return das in ein java.awt.Rectangle-Objekt umgewandelte Rechteck
	 */
	public java.awt.Rectangle toAWTRectangle() {
		Play.getPlay().getActivePerformance().checkStop();
		Performance.getPerformance().lock();
		try {
			return new java.awt.Rectangle(this.x, this.y, this.width,
					this.height);
		} finally {
			Performance.getPerformance().unlock();
		}
	}
}
