import theater.*;

import java.util.*;

import javax.swing.*;


/**
 * Die Klasse stellt eine Repraesentation des virtuellen Schreibers dar.
 *
 * @author Dietrich Boles (Universitaet Oldenburg)
 * @version 1.0 (23.10.2009)
 *
 */
public class Schreiber extends Actor {
    protected Solist out;
    protected Solist System;
    protected Solist IO;
    protected int currentRow;
    protected int currentCol;
    protected String input;
    protected TheaterImage image;

    // Default-Konstruktor (muss implementiert werden!)
    public Schreiber() {
        super(); // nicht loeschen!
        updateImage();

        // hiermit kann dem Solist ein "Name" gegeben werden
        out = (Solist) this;
        System = out;
        IO = out;

        // weitere Initialisierungen
        currentRow = 0;
        currentCol = 0;
    }

    // Copy-Konstruktor (muss implementiert werden!)
    public Schreiber(Schreiber actor) {
        super(actor); // nicht loeschen!

        // hiermit kann dem Solist ein "Name" gegeben werden
        out = (Solist) this;
        System = out;
        IO = out;

        // weitere Initialisierungen
        currentRow = actor.currentRow;
        currentCol = actor.currentCol;
    }

    @Description("Liefert die Anzahl an Spalten des Terminals.")
    public int getColumns() {
        Performance.getPerformance().lock();

        try {
            return Terminal.COLS;
        } finally {
            Performance.getPerformance().unlock();
        }
    }

    @Description("Liefert die Anzahl an Reihen des Terminals.")
    public int getRows() {
        Performance.getPerformance().lock();

        try {
            return Terminal.ROWS;
        } finally {
            Performance.getPerformance().unlock();
        }
    }

    @Description("<html>Verschiebt den Cursor an die angegebene Position " +
    "(erster Parameter = Reihe, zweiter Parameter = Spalte)." +
    "<br>Wenn ungltige Koordinaten bergeben werden, passiert nichts.</html>")
    public void move(int row, int col) {
        Performance.getPerformance().lock();

        try {
            if ((0 <= row) && (row < Terminal.ROWS) && (0 <= col) &&
                    (col < Terminal.COLS)) {
                currentRow = row;
                currentCol = col;
                setLoc();
            }
        } finally {
            Performance.getPerformance().unlock();
        }
    }

    @Description("Lscht das Terminal.")
    public void clear() {
        Performance.getPerformance().lock();
        Performance.getPerformance().freeze();

        try {
            getStage().remove(getStage().getComponents(Zeichen.class));
        } finally {
            Performance.getPerformance().unfreeze();
            Performance.getPerformance().unlock();
        }
    }

    @Description("Gibt das bergeben Zeichen aus.")
    public void print(char zeichen) {
        Performance.getPerformance().lock();
        Performance.getPerformance().freeze();

        try {
            List<Component> z = getStage()
                                    .getComponentsAt(currentCol, currentRow,
                    Zeichen.class);

            if (z.size() == 0) {
                getStage().add(new Zeichen(zeichen), currentCol, currentRow);
            } else {
                ((Zeichen) z.get(0)).set(zeichen);
            }

            currentCol++;

            if (currentCol == Terminal.COLS) {
                currentCol = 0;
                currentRow++;

                if (currentRow == Terminal.ROWS) {
                    currentRow = 0;
                }
            }
            setLoc();
        } finally {
            Performance.getPerformance().unfreeze();
            Performance.getPerformance().unlock();
        }
    }

    @Description("Gibt die bergebene Zeichenkette aus.")
    public void print(String str) {
        Performance.getPerformance().lock();
        Performance.getPerformance().freeze();

        try {
            for (int i = 0; i < str.length(); i++) {
                print(str.charAt(i));
            }
        } finally {
            Performance.getPerformance().unfreeze();
            Performance.getPerformance().unlock();
        }
    }

    @Description("Gibt eine String-Reprsentation des bergebenen Objektes aus.")
    public void print(Object obj) {
        Performance.getPerformance().lock();

        try {
            print(obj.toString());
        } finally {
            Performance.getPerformance().unlock();
        }
    }

    @Description("Gibt das bergeben Zeichen aus und verschiebt den Cursor an den Anfang der nchsten Zeile.")
    public void println(char zeichen) {
        Performance.getPerformance().lock();

        try {
            print(zeichen);
            newLine();
        } finally {
            Performance.getPerformance().unlock();
        }
    }

    @Description("Gibt die bergebene Zeichenkette aus und verschiebt den Cursor an den Anfang der nchsten Zeile.")
    public void println(String str) {
        Performance.getPerformance().lock();
        Performance.getPerformance().freeze();

        try {
            print(str);
            newLine();
        } finally {
            Performance.getPerformance().unfreeze();
            Performance.getPerformance().unlock();
        }
    }

    @Description("Gibt eine String-Reprsentation des bergebenen Objektes aus und verschiebt den Cursor an den Anfang der nchsten Zeile.")
    public void println(Object obj) {
        Performance.getPerformance().lock();

        try {
            print(obj);
            newLine();
        } finally {
            Performance.getPerformance().unlock();
        }
    }

    @Description("Verschiebt den Cursor an den Anfang der nchsten Zeile.")
    public void println() {
        Performance.getPerformance().lock();

        try {
            newLine();
        } finally {
            Performance.getPerformance().unlock();
        }
    }

    @Description("<html>Gibt die bergebene Zeichenkette aus und wartet auf die " +
    "Eingabe einer Zeichenkette durch den Nutzer. " +
    "<br>Nach Drcken der Enter-Taste wird die eingegebene Zeichenkette " +
    "als Wert geliefert.</html>")
    public String readString(String prompt) {
        print(prompt);

        // Falls der Befehl ber das Befehlsfenster ausgefhrt wird.
        if (!((TPerformance) Performance.getPerformance()).isRunning()) {
            return "";
        }

        synchronized (this) {
            input = "";
            getStage().setHandlingKeyEvents(true);

            try {
                wait();
            } catch (InterruptedException exc) {
            }

            getStage().setHandlingKeyEvents(false);
        }

        return input;
    }

    @Description("<html>Gibt die bergebene Zeichenkette aus und wartet auf die " +
    "Eingabe eines int-Wertes durch den Nutzer. " +
    "<br>Nach Drcken der Enter-Taste wird der eingegebene Wert geliefert." +
    "<br>Wenn der Nutzer einen ungltigen int-Wert eingibt, wird 0 geliefert.</html>")
    public int readInt(String prompt) {
        String input = readString(prompt);

        try {
            int res = Integer.parseInt(input);

            return res;
        } catch (NumberFormatException exc) {
            return 0;
        }
    }

    @Description("<html>Gibt die bergebene Zeichenkette aus und wartet auf die " +
    "Eingabe eines short-Wertes durch den Nutzer. " +
    "<br>Nach Drcken der Enter-Taste wird der eingegebene Wert geliefert." +
    "<br>Wenn der Nutzer einen ungltigen short-Wert eingibt, wird 0 geliefert.</html>")
    public short readShort(String prompt) {
        String input = readString(prompt);

        try {
            short res = Short.parseShort(input);

            return res;
        } catch (NumberFormatException exc) {
            return 0;
        }
    }

    @Description("<html>Gibt die bergebene Zeichenkette aus und wartet auf die " +
    "Eingabe eines long-Wertes durch den Nutzer. " +
    "<br>Nach Drcken der Enter-Taste wird der eingegebene Wert geliefert." +
    "<br>Wenn der Nutzer einen ungltigen long-Wert eingibt, wird 0L geliefert.</html>")
    public long readLong(String prompt) {
        String input = readString(prompt);

        try {
            long res = Long.parseLong(input);

            return res;
        } catch (NumberFormatException exc) {
            return 0L;
        }
    }

    @Description("<html>Gibt die bergebene Zeichenkette aus und wartet auf die " +
    "Eingabe eines float-Wertes durch den Nutzer. " +
    "<br>Nach Drcken der Enter-Taste wird der eingegebene Wert geliefert." +
    "<br>Wenn der Nutzer einen ungltigen float-Wert eingibt, wird 0.0F geliefert.</html>")
    public float readFloat(String prompt) {
        String input = readString(prompt);

        try {
            float res = Float.parseFloat(input);

            return res;
        } catch (NumberFormatException exc) {
            return 0.0F;
        }
    }

    @Description("<html>Gibt die bergebene Zeichenkette aus und wartet auf die " +
    "Eingabe eines double-Wertes durch den Nutzer. " +
    "<br>Nach Drcken der Enter-Taste wird der eingegebene Wert geliefert." +
    "<br>Wenn der Nutzer einen ungltigen double-Wert eingibt, wird 0.0 geliefert.</html>")
    public double readDouble(String prompt) {
        String input = readString(prompt);

        try {
            double res = Double.parseDouble(input);

            return res;
        } catch (NumberFormatException exc) {
            return 0.0;
        }
    }

    @Description("<html>Gibt die bergebene Zeichenkette aus und wartet auf die " +
    "Eingabe eines Zeichens durch den Nutzer. " +
    "<br>Nach Drcken der Enter-Taste wird das eingegebene Zeichen geliefert." +
    "<br>Wenn der Nutzer kein Zeichen eingibt, wird '\\0' geliefert.</html>")
    public char readChar(String prompt) {
        String input = readString(prompt);

        try {
            char res = input.charAt(0);
          
            return res;
        } catch (Exception exc) {
            return '\0';
        }
    }

    @Description("<html>Gibt die bergebene Zeichenkette aus und wartet auf die " +
    "Eingabe eines boolean-Wertes (true, false) durch den Nutzer. " +
    "<br>Nach Drcken der Enter-Taste wird der eingegebene Wert geliefert." +
    "<br>Wenn der Nutzer einen ungltigen boolean-Wert eingibt, wird 'false' geliefert.</html>")
    public boolean readBoolean(String prompt) {
        String input = readString(prompt);

        try {
            boolean res = Boolean.parseBoolean(input);

            return res;
        } catch (NumberFormatException exc) {
            return false;
        }
    }

    @Invisible
    void newInput(char zeichen) {
        print(zeichen);
        input = input + zeichen;
    }

    @Invisible
    void delInput() {
        if (input.length() > 0) {
            input = input.substring(0, input.length() - 1);
            currentCol--;

            if (currentCol < 0) {
                currentCol = Terminal.COLS - 1;
                currentRow--;

                if (currentRow < 0) {
                    currentRow = Terminal.ROWS - 1;
                }
            }
            setLoc();

            getStage()
                .remove(getStage()
                            .getComponentsAt(currentCol, currentRow,
                    Zeichen.class));
        }
    }

    @Invisible
    void endInput() {
        println();
    }

    @Invisible
    private void newLine() {
        currentCol = 0;
        currentRow++;

        if (currentRow == Terminal.ROWS) {
            currentRow = 0;
        }
        setLoc();
    }
    
    @Invisible
    private void setLoc() {
        setLocation(currentCol, currentRow);
    }
    
    @Invisible
    void updateImage() {
        image = new TheaterImage(Terminal.CELLSIZE, Terminal.CELLSIZE);
        image.setColor(Terminal.BG_COLOR);
        image.fill();
        image.setColor(Terminal.FG_COLOR);
        image.drawLine(0, Terminal.CELLSIZE - 1, Terminal.CELLSIZE, Terminal.CELLSIZE - 1);
        setImage(image);
    }
}
