package util;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.util.Locale;
import java.util.Properties;

/**
 * Verwaltung der Properties; aktuell language und jdk.
 * 
 * @author Dietrich Boles, Uni Oldenburg
 * @version 1.0 (12.11.2008)
 * 
 */
public class PropertyManager {

	public static String PROP_FILE = "solist.properties";

	public static String PROP_LANGUAGE = "language";
	public static String PROP_JDK = "jdk";
	public static String PROP_DEBUGGING = "debugging";
	public static String PROP_INITHEAPSPACE = "initheapspace";
	public static String PROP_MAXHEAPSPACE = "maxheapspace";
	public static String PROP_HANDBOOK_ONLINE = "handbookonline";
	public static String PROP_HANDBOOK_OFFLINE = "handbookoffline";

	private static String DEF_LANGUAGE = "de"; // de / en
	private static String DEF_JDK = null;
	private static String DEF_DEBUGGING = "on"; // on / off
	private static String DEF_INITHEAPSPACE = "32"; // -Xms32m, Standard ist 2M,
	// min 1M
	private static String DEF_MAXHEAPSPACE = "128"; // -Xmx128m, Standard ist
	// 64M, min 2M

	private static String DEF_HANDBOOK_ONLINE = null; // URL
	private static String DEF_HANDBOOK_OFFLINE = null; // Dateiname
	

	// siehe http://java.sun.com/j2se/1.5.0/docs/tooldocs/solaris/java.html

	private static PropertyManager manager = null;
	private Properties props;

	/**
	 * erzeugt einen Property-Manager. Geladen werden die Properties aus der
	 * Datei "solist.properties"
	 */
	protected PropertyManager() {
		try {
			this.props = new Properties();
			if (new File(PropertyManager.PROP_FILE).exists()) {
				FileInputStream propInFile = new FileInputStream(
						PropertyManager.PROP_FILE);
				this.props.load(propInFile);
				propInFile.close();
			}
		} catch (Throwable exc) {
			exc.printStackTrace();
		}
	}

	/**
	 * Liefert den PropertyManager
	 * 
	 * @return
	 */
	public static PropertyManager getPropertyManager() {
		if (PropertyManager.manager == null) {
			PropertyManager.manager = new PropertyManager();
		}
		return PropertyManager.manager;
	}

	/**
	 * Liefert die eingestellte Sprache (deutsch oder englisch)
	 * 
	 * @return
	 */
	public String getLanguage() {
		String lang = this.props.getProperty(PROP_LANGUAGE);
		return lang != null ? lang.trim() : PropertyManager.DEF_LANGUAGE;
	}

	/**
	 * Liefert die Locale in Abhngigkeit der eingestellten Sprache
	 * 
	 * @return
	 */
	public Locale getLocale() {
		String language = this.getLanguage();
		if (language.equals("en")) {
			return Locale.ENGLISH;
		} else if (language.equals("de")) {
			return Locale.GERMAN;
		}
		return Locale.getDefault();
	}

	/**
	 * Speichert die Property jdk. Hierin wird gespeichert, welches JDK benutzt
	 * werden soll.
	 * 
	 * @param jdk
	 */
	public void setJDK(String jdk) {
		this.props.setProperty(PROP_JDK, jdk);
		this.store();
	}

	/**
	 * Liefert die Property jdk. Hierin ist gespeichert, welches JDK benutzt
	 * werden soll. Es kann auch null geliefert werden, wenn die Property noch
	 * nicht gesetzt wurde.
	 * 
	 * @return
	 */
	public String getJDK() {
		String jdk = this.props.getProperty(PROP_JDK);
		if (jdk == null || jdk.trim().equals("")) {
			return DEF_JDK;
		} else {
			return jdk.trim();
		}
	}

	public String getInitHeapSpaceP() {
		String space = this.props.getProperty(PROP_INITHEAPSPACE);
		if (space == null) {
			return DEF_INITHEAPSPACE;
		} else {
			return space.trim();
		}
	}

	public String getMaxHeapSpaceP() {
		String space = this.props.getProperty(PROP_MAXHEAPSPACE);
		if (space == null) {
			return DEF_MAXHEAPSPACE;
		} else {
			return space.trim();
		}
	}

	public String getInitHeapSpace() {
		String space = this.props.getProperty(PROP_INITHEAPSPACE);
		if (space == null) {
			return "-Xms" + DEF_INITHEAPSPACE + "m";
		} else {
			return "-Xms" + space.trim() + "m";
		}
	}

	public String getMaxHeapSpace() {
		String space = this.props.getProperty(PROP_MAXHEAPSPACE);
		if (space == null) {
			return "-Xmx" + DEF_MAXHEAPSPACE + "m";
		} else {
			return "-Xmx" + space.trim() + "m";
		}
	}

	public String getHandbookOnline() {
		String url = this.props.getProperty(PROP_HANDBOOK_ONLINE);
		if (url == null || url.trim().equals("")) {
			return DEF_HANDBOOK_ONLINE;
		} else {
			return url.trim();
		}
	}

	public String getHandbookOffline() {
		String url = this.props.getProperty(PROP_HANDBOOK_OFFLINE);
		if (url == null || url.trim().equals("")) {
			return DEF_HANDBOOK_OFFLINE;
		} else {
			return url.trim();
		}
	}

	public boolean isDebuggingOn() {
		String value = this.props.getProperty(PROP_DEBUGGING);
		if (value == null) {
			return DEF_DEBUGGING.equals("on");
		} else {
			return value.trim().equals("on");
		}
	}

	public void store() {
		try {
			FileOutputStream prop = new FileOutputStream(
					PropertyManager.PROP_FILE);
			this.props.store(prop, "");
			prop.close();
		} catch (Throwable th) {
			th.printStackTrace();
		}
	}

}
