package scratch;

import java.util.ArrayList;

import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamWriter;

import model.Play;

import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

import scratch.elements.RootBooleanObject;
import scratch.elements.RootVoidObject;
import scratch.gui.TabButton;
import scratch.gui.TabPanel;

/**
 * Die Klasse Method speichert alle Inhalte einer Methode. Die komplette
 * Definition einer Methode und die zu- stzlichen Inhalte, die im
 * Bearbeitungsfenster angezeigt werden, jedoch nicht fr die Ausfhrung
 * notwendig sind.
 * 
 * @author HackZ
 * 
 */
public class ScratchMethod {
	private int top = 0;
	private int left = 0;
	private Renderable rootElement;
	private ArrayList<Renderable> renderables;
	private String name;
	private Renderable.Type type;
	private boolean opened;
	
	public static String selectedMethod = null;

	/**
	 * Erstelle eine neue Methode mit dem bergebenen Namen und Typ
	 * 
	 * @param name
	 *            Name der Methode
	 * @param type
	 *            Typ der Methode (VOID, BOOLEAN, ...)
	 */
	public ScratchMethod(String name, Renderable.Type type) {
		this.name = name;
		this.type = type;
		renderables = new ArrayList<Renderable>();
		if (type == Renderable.Type.BOOLEAN)
			rootElement = new RootBooleanObject();
		else
			rootElement = new RootVoidObject();
		rootElement.moveTo(ScratchPanel.OPTIONS_WIDTH + 20,
				ScratchPanel.TAB_PANEL_HEIGHT);
	}

	/**
	 * Liefert das erste ausfhrbare Renderable Objekt dieser Methode
	 * 
	 * @return
	 */
	public Renderable getRootElement() {
		return rootElement;
	}

	/**
	 * Liefert alle nicht ausfhrbaren Renderstacks dieser Methode
	 * 
	 * @return
	 */
	public ArrayList<Renderable> getRenderables() {
		return renderables;
	}

	/**
	 * Liefert die y-Achsenverschiebung des Fensters dieser Methode
	 * 
	 * @return
	 */
	public int getTop() {
		return top;
	}

	/**
	 * Liefert die x-Achsenverschiebung des Fensters dieser Methode
	 * 
	 * @return
	 */
	public int getLeft() {
		return left;
	}

	/**
	 * Setzt die y-Achsenverschiebung des Fensters dieser Methode
	 * 
	 * @param top
	 */
	public void setTop(int top) {
		this.top = top;
	}

	/**
	 * Setzt die x-Achsenverschiebung des Fensters dieser Methode
	 * 
	 * @param left
	 */
	public void setLeft(int left) {
		this.left = left;
	}

	/**
	 * Liefert den Namen dieser Methode
	 * 
	 * @return
	 */
	public String getName() {
		return name;
	}

	/**
	 * Liefert den Typ dieser Methode
	 * 
	 * @return
	 */
	public Renderable.Type getType() {
		return type;
	}

	/**
	 * Liefert, ob der Tab dieser Methode geffnet ist
	 * 
	 * @return
	 */
	public boolean isOpened() {
		return opened;
	}

	/**
	 * Setzt fest, ob der Tab dieser Methode geffnet ist
	 * 
	 * @param opened
	 */
	public void setOpened(boolean opened) {
		this.opened = opened;
	}
	
	/**
	 * Fragt ab, ob das Element mit dem bergebenen Namen in dieser Methode in
	 * irgendeiner Art verwendet wird. Dies ist notwendig, um das Element zu
	 * lschen, denn es darf nirgendwo verwendet worden sein.
	 * 
	 * @param name
	 *            Name des Elements, nach dem gesucht werden soll
	 * @return
	 */
	public boolean inUse(String name) {
		for (Renderable r : renderables)
			if (r.inUse(name))
				return true;

		if (rootElement.inUse(name))
			return true;

		return false;
	}

	/**
	 * Benennt die Methode mit dem Namen fromName in den Namen toName. Diese
	 * Methode ruft rekursiv die Methoden der Kinder auf.
	 * 
	 * @param fromName
	 * @param toName
	 */
	public void rename(String fromName, String toName) {
		if (this.name.equals(fromName))
			this.name = toName;

		for (Renderable r : renderables)
			r.rename(fromName, toName);

		rootElement.rename(fromName, toName);
	}

	/**
	 * Schreibt den Inhalt der Methode in ein XML-Document. Das XML-Document
	 * muss von ausserhalb bereits vorbereitet worden sein, mit dem
	 * Startelement.
	 * 
	 * @param writer
	 *            Der writer, in dem das XML aufgebaut wird.
	 * @throws XMLStreamException
	 */
	public void toXML(XMLStreamWriter writer) throws XMLStreamException {
		writer.writeStartElement("METHOD");
		writer.writeAttribute("NAME", name);
		writer.writeAttribute("TYPE", type.toString());
		writer.writeAttribute("OPENED", opened ? "T" : "F");
		TabPanel p = Play.getPlay().getPlayFrame().getScratchPanel()
				.getTabPanel();
		TabButton b = p.getTab(name);
		if (b != null) {
			writer.writeAttribute("SELECTED", b.isSelected() ? "T" : "F");
		} else {
			writer.writeAttribute("SELECTED", "F");
		}

		// Root Element schreiben
		writer.writeStartElement("ROOTELEMENT");
		Renderable root = rootElement.next;
		if (root != null)
			root.toXML(writer);
		writer.writeEndElement();

		// Renderables schreiben
		for (Renderable r : renderables) {
			writer.writeStartElement("RENDERSTACK");
			writer.writeAttribute("TOP", r.getY() + "");
			writer.writeAttribute("LEFT", r.getX() + "");
			r.toXML(writer);
			writer.writeEndElement();
		}

		writer.writeEndElement();
	}

	/**
	 * Ldt die Methode anhand der bergebenen NodeList im DOM-Baum. Dafr
	 * mssen alle Kinderknoten des Elements <tt>METHOD</tt> bergeben werden
	 * 
	 * @param childNodes
	 */
	public void loadProgram(NodeList childNodes) {
		// RootElement laden
		NodeList rootNodes = getRootNodeList(childNodes);
		if (rootNodes != null && rootNodes.getLength() > 0) {
			Element rootElem = (Element) rootNodes.item(0);
			Renderable rootRenderable = ScratchUtils.getRenderableByName(
					rootElem.getAttribute("NAME"), rootElem
							.getAttribute("TYPE"));
			rootElement.addAsNext(rootRenderable);
			rootRenderable.loadProgram(rootElem.getChildNodes());
			rootRenderable.updateBounds();
			rootRenderable.updateChilds();
		}

		// RenderStacks laden
		for (int i = 0; i < childNodes.getLength(); i++) {
			Element child = (Element) childNodes.item(i);
			if (!child.getLocalName().equals("RENDERSTACK"))
				continue;

			loadRenderstack(child);
		}
	}

	/**
	 * Findet in den bergebenen Kinbderelementen eines DOM-Baums, das Element
	 * <tt>ROOTELEMENT</tt> und gibt dessen Kinderelemente zurck.
	 * 
	 * @param childNodes
	 *            Kinderelemente von <tt>METHOD</tt>
	 * @return Kinderelemente von <tt>ROOTELEMENT</tt>
	 */
	private NodeList getRootNodeList(NodeList childNodes) {
		for (int i = 0; i < childNodes.getLength(); i++)
			if (childNodes.item(i).getLocalName().equals("ROOTELEMENT"))
				return childNodes.item(i).getChildNodes();

		return null;
	}

	/**
	 * Ldt alle Elemente des Renderstacks
	 * 
	 * @param child
	 *            ein <tt>RENDERSTACK</tt> aus <tt>METHOD</tt>
	 */
	private void loadRenderstack(Element child) {
		int posX = new Integer(child.getAttribute("LEFT"));
		int posY = new Integer(child.getAttribute("TOP"));
		Element childNodes = (Element) child.getChildNodes().item(0);
		Renderable childRenderable = ScratchUtils.getRenderableByName(
				childNodes.getAttribute("NAME"), childNodes
						.getAttribute("TYPE"));
		childRenderable.loadProgram(childNodes.getChildNodes());
		childRenderable.updateBounds();
		childRenderable.getRootElement().updateChilds();

		childRenderable.moveTo(posX, posY);
		renderables.add(childRenderable);
	}
}
